#!/usr/bin/perl
# IBM_PROLOG_BEGIN_TAG 
# This is an automatically generated prolog. 
#  
#  
#  
# Licensed Materials - Property of IBM 
#  
# (C) COPYRIGHT International Business Machines Corp. 2003,2004 
# All Rights Reserved 
#  
# US Government Users Restricted Rights - Use, duplication or 
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp. 
#  
# IBM_PROLOG_END_TAG 
#
# @(#)31   1.35.1.12   src/csm/core/pm/NetworkUtils.pm.perl, setup, csm_rameh, rameh_base 6/23/04 17:51:15
#
#####################################################################

package NetworkUtils;

use strict;

#            locale tells perl to honor locale for sorting, dates, etc.
#            More info about locale.pm and perl handling of locales can be found in
#            /usr/lib/perl5/5.6.0/locale.pm and the man page for perllocale.

use locale;
use Socket;
use File::Find;

my $msgs;
my $distro;
my $useTranslatedMsg;
my %catHashes;
my $csmroot;
my ($MSGCAT, $MSGMAPPATH, $MSGSET);
my $NO_NODERANGES;
my $NODEGROUPEXPMEM_WARNING = 1;

# $NodeUtils::errno;          # Will be set if an error occurs.  You must zero
# this out before calling a NodeUtils function,
# if you want to check it afterwards.

BEGIN
{

    #    This enables us to redirect where it looks for other CSM files during development
    $csmroot = $ENV{'CSM_ROOT'} ? $ENV{'CSM_ROOT'} : '/opt/csm';

    $MSGCAT = 'nodecmds.cat';
    if (defined $::MSGMAPPATH)
    {
        $MSGMAPPATH = $ENV{'CSM_ROOT'} ? "$csmroot/msgmaps" : $::MSGMAPPATH;
    }
    else
    {
        $MSGMAPPATH = "$csmroot/msgmaps";
    }
    $MSGSET = 'NodeUtils';
}

umask(0022);   #  This sets umask for all CSM files so that group and world only
               #  have read permissions.
               #  To change it, simply use the umask call in your script, after
               #  the "use NetworkUtils;" line

#--------------------------------------------------------------------------------

=head1    NetworkUtils


This program module file, supports the CSM/install networking dependencies;  
from datalink to session layers.



If adding to this file, please take a moment to ensure that:

    1.  Your contrib has a readable pod header describing the purpose and use of
         the subroutine.

    2. Your contrib is under the correct heading and is in alphabetical order
    under that heading.

    3. You  have run tidypod on your this file and copied NetworkUtils.pm.perl.tdy to
       NetworkUtils.pm.perl before checking in

       you can examine the pod output in  NetworkUtils.pm.perl.html file in a browser.

    4. Copy the new ./NetworkUtils.pm.perl.html to /project/csm/www, to update the pod.
       for the team.

=cut

#--------------------------------------------------------------------------------

=head2    DHCP Support

=cut

#--------------------------------------------------------------------------------

=head3    verify_dhcp

        Check that dhcp is up and running.

        Arguments:
                none
        Returns:
                0 - dhcp is running
                numErrors from attempt to start dhcp
        Globals:
                none
        Error:
                numErrors from attempt to start dhcp
        Example:
                $rc = NetworkUtils->verify_dhcp();
        Comments:
                Right now we don't check the /etc/dhcpd.conf file, since that
                should have  been verified when it was created.  This function
                is just for extremely paranoid peopole.

=cut

#--------------------------------------------------------------------------------

sub verify_dhcp
{
    my $errors = 0;
    my @output;
    my $cmd;
    NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                              'V', 'IMsgDhcpCheck');
    unless (-f "/var/lib/dhcp/dhcpd.leases")
    {
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'I', 'IMsgDhcpMissing');
        unless (-d "var/lib/dhcp")
        {
            NodeUtils->runcmd("$::MKDIR -p /var/lib/dhcp");
        }
        NodeUtils->runcmd("$::TOUCH /var/lib/dhcp/dhcpd.leases");
    }

    # SuSE Change
    #$cmd = "$::SERVICE dhcpd status";
    $cmd = NodeUtils->service("dhcpd", "status");

    chomp(@output = `$cmd`);
    my @line = split(" ", @output);
    if (grep /running/, @output)
    {
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'V', 'IMsgDhcpOutput', @output);
    }
    else
    {
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'V', 'IMsgDhcpOutput', @output);
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'V', 'IMsgDhcpProblemRestart', @output);

        # SuSE Change
        #`$::SERVICE dhcpd start`;
        #$cmd = "$::SERVICE dhcpd status";
        $cmd = NodeUtils->service("dhcpd", "start");
        `$cmd`;
        $cmd = NodeUtils->service("dhcpd", "status");

        chomp(@output = `$cmd`);
        if (grep /running/, @output)
        {

            #print "dhcp: @output\n" if $::VERBOSE;
            NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'V', 'IMsgDhcpOutput',
                                      @output);
        }
        else
        {
            NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'I', 'IMsgDhcpNotWorking');
            $errors++;
        }
    }
    return $errors;
}

#--------------------------------------------------------------------------------

=head2    Ethernet Support

=cut

#--------------------------------------------------------------------------------

=head3    chkethup

         Ensure that the eth0 is up and running 

        Arguments:
                none
        Returns:
                none
        Globals:
                none
        Error:
                messageFromCat E1
        Example:
                NetworkUtils->chkethup();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub chkethup
{
    my $cmd = "$::NETSTAT -rn | $::GREP eth0";
    unless (`$cmd`)
    {
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'E1', 'EMsgEth0NotUp');
    }
    return;
}

#--------------------------------------------------------------------------------

=head2    IP Address Support

=cut

#--------------------------------------------------------------------------------

=head3    genHex

        Generates a hex representation of an ip address:
            for example, 9.114.133.193 --> 097285C1

        Takes in a list of hosts and returns a hash where the key is
        the hostname and the value is the IP in hex form. 

        Arguments:
                @hostNames
        Returns:
                Hash of hostName / IP address (in hex) pairs.
        Globals:
                none
        Error:
                undefined
        Example:
                %hex = NetworkUtils->genHex(@::nodes);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub genHex
{
    shift;
    my $host;
    my $uno;
    my $dos;
    my $tres;
    my $cuatro;
    my $hex;
    my %hex;

    foreach $host (@_)
    {
        my $ip = NodeUtils->getHost($host);    # Get ip address
        ($uno, $dos, $tres, $cuatro) = split(/\./, $ip);    # Split by the .
        $hex = sprintf("%02X", $uno);    # put formatted into hex
        $hex .= sprintf("%02X", $dos);
        $hex .= sprintf("%02X", $tres);
        $hex .= sprintf("%02X", $cuatro);
        $hex{$host} = $hex;
    }
    return %hex;
}

#--------------------------------------------------------------------------------

=head3    inc_ip

        Increment an IP address by 1, carrying to the previous octet as required.
        This code originally appeared in the LUI product (luiapi.pm)

        Arguments:
                $IPaddress
        Returns:
                IP address as a string
        Globals:
                none
        Error:
                csmInstall.cat E
        Example:
                $node = NetworkUtils->inc_ip($ipaddr); 
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub inc_ip
{
    my $ip = $_[1];
    my ($a, $b, $c, $d);
    ($a, $b, $c, $d) = split(/\./, $ip);
    $d++;
    if ($d > 255)
    {
        $d = $d - 256;
        $c++;
        if ($c > 255)
        {
            $c = $c - 256;
            $b++;
            if ($b > 255)
            {
                $b = $b - 256;
                $a++;
                if ($a > 255)
                {
                    NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                          'csminstall', 'E', 'EMsgIPADDR_ERROR', "$a.$b.$c.$d");

                    #printf "Error - ip address out of valid range.\n";
                    return 1;
                }
            }
        }
    }
    return $a . "\." . $b . "\." . $c . "\." . $d;

}

#--------------------------------------------------------------------------------

=head3    isIpaddr

    returns 1 if parameter is has a valid IP address form.

    Arguments:
        dot qulaified IP address: e.g. 1.2.3.4
    Returns:
        1 - if legal IP address
        0 - if not legal IP address.
    Globals:
        none
    Error:
        none
    Example:
         if (NetworkUtils->isIpaddr( $ipAddr )) { blah; }
    Comments:
        Doesn't test if the IP address is on the network,
        just tests its form.

=cut

#--------------------------------------------------------------------------------

sub isIpaddr
{
    my ($class, $addr) = @_;

    #print "addr=$addr\n";
    if ($addr =~ /^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$/)
    {
        return 1;
    }
    else { return 0; }
}

#--------------------------------------------------------------------------------

=head3    subnet

        Returns the subnet in dot format

        Arguments:
                $hostName
                $netmask
        Returns:
                subnet for $hostName
        Globals:
                none
        Error:
                none
        Example:
                never used.
        Comments:
                never used???

=cut

#--------------------------------------------------------------------------------

sub subnet
{
    my ($class, $host, $netmask) = @_;

    #$host = `hostname` if (!$host);
    #$netmask = '255.255.255.0' if (!$netmask);

    chomp $host;
    chomp $netmask;

    #print "INPUT:  $host $netmask\n";

    my ($hostname, $ipaddr) = NodeUtils->getHost($host);

    my ($ia, $ib, $ic, $id) = split('\.', $ipaddr);
    my ($na, $nb, $nc, $nd) = split('\.', $netmask);

    # Convert to integers so the bitwise and (&) works correctly.
    int $ia;
    int $ib;
    int $ic;
    int $id;
    int $na;
    int $nb;
    int $nc;
    int $nd;

    my $sa     = ($ia & $na);
    my $sb     = ($ib & $nb);
    my $sc     = ($ic & $nc);
    my $sd     = ($id & $nd);
    my $subnet = "$sa.$sb.$sc.$sd";

    #print "IPADDR = $ia.$ib.$ic.$id\n";
    #print "NETMSK = $na.$nb.$nc.$nd\n";
    #print "SUBNET = $subnet\n";

    return $subnet;
}

#--------------------------------------------------------------------------------

=head2    MAC Address Support

=cut

#--------------------------------------------------------------------------------

=head3    getmacs_via_dsh

        Gathers MAC addresses for an array of node Names.

        Arguments:
                referance to an array of Node Names
        Returns:
                Array of unreachable node names
        Globals:
                none
        Error:
                none
        Example:
                unused

        Comments:
                unused???

=cut

#--------------------------------------------------------------------------------

sub getmacs_via_dsh
{
    my ($class, $ref_nodelist) = @_;
    my @nodelist = @$ref_nodelist;
    NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                              'I', 'IMsgGETTING_MACS_VIA_DSH');
    my $dshcmd =
      '/sbin/ifconfig eth0 | /bin/grep HWaddr | /bin/awk "{print \$5}"';
    my $RemoteShell = InstallUtils->getRemoteShell();
    my @new_getmacslist;
    my @unreach_nodes = ();
    my @reach         = ();

    my $node_file =
      InstallUtils->make_node_list_file(\@nodelist)
      ;    #put all dest nodes in tmp file
    my $cmd = "$::REMOTESHELL_EXPECT -d $node_file";
    NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                              "V", 'IMsgRUN_CMD', $cmd);
    my @output = `$cmd 2>&1`;
    InstallUtils->close_delete_file($::NODE_LIST_FILE, $node_file)
      ;    #delete node file
    my %out;

    foreach my $line (@output)
    {
        chomp $line;
        my @parts = split ":", $line;
        if ($parts[1] =~ m/test.success/)
        {    #this node is reachable
            $out{$parts[0]} = 1;
        }
    }
    foreach my $n (@nodelist)
    {
        if (!$out{$n})
        {
            push @unreach_nodes, $n;
        }
        else
        {    #it is reachable
            push @reach, $n;
        }
    }
    if (@unreach_nodes)
    {
        my $node_str = join ', ', @unreach_nodes;
        NodeUtils->messageFromCat(
                                  'csmInstall.cat',       $::MSGMAPPATH,
                                  'csminstall',           'W',
                                  'EMsgDSHCannotConnect', $RemoteShell,
                                  $node_str
                                  );
        push @new_getmacslist, @unreach_nodes;
    }
    if (@reach)
    {

        #we can theoretically reach it with DSH
        $node_file =
          InstallUtils->make_node_list_file(\@reach)
          ;    #put all dest nodes in tmp file
        $ENV{'DSH_LIST'} = $node_file;
        my $output = NodeUtils->runcmd("$::DSH '$dshcmd'", -1, 1);
        InstallUtils->close_delete_file($::NODE_LIST_FILE, $node_file)
          ;    #delete node file
        my %macs;
        foreach my $line (@$output)
        {
            chomp $line;
            my ($nodename, $macaddr) = split(': ', $line);
            if ($macaddr =~ /(..:..:..:..:..:..)/)
            {
                $macs{$nodename} = $macaddr;
            }
        }
        foreach my $node (@reach)
        {
            if ($macs{$node})
            {
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'I', 'IMsgFOUND_MACADDR', $node,
                                  $macs{$node});
                NodeUtils->runcmd(
                          "$::CHNODE $node InstallAdapterMacaddr=$macs{$node}");
            }
            else
            {
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                 'csminstall', 'I', 'IMsgCANT_DSH_NODE', $node);
                push(@new_getmacslist, $node);
            }
        }
    }
    return (@new_getmacslist);
}

#--------------------------------------------------------------------------------

=head2    NFS Support

=cut

#--------------------------------------------------------------------------------

=head3   cacheEtcHosts

    This subroutine caches /etc/hosts to improve performance for large clusters.

    Arguments:
        none
    Returns:
        none
    Globals:
        none
    Error:
        none
    Example:
        none
    Comments:
        This routine is only called if more than 15 nodes are used.

=cut

#--------------------------------------------------------------------------------

sub cacheEtcHosts
{
    if (defined $NodeUtils::cacheetchosts)
    {
        return;
    }
    else
    {
        $NodeUtils::cacheetchosts = 1;    #so we don't do this again
    }
    my $cache = 0;
    if (NodeUtils->isLinux())
    {

        #On linux you can tell if /etc/hosts is used by looking at the /etc/host.conf
        my $file = "/etc/host.conf";
        if (open(CONF, $file))
        {
            while (<CONF>)
            {
                my $line = $_;
                chomp $line;
                if ($line =~ m/^\s*order\s+hosts/)
                {
                    $cache = 1;
                    last;
                }
            }
            close(CONF);
        }
        else { return; }

    }
    elsif (NodeUtils->isAIX())
    {

        #there is an environement variable
        if (defined $ENV{'NSORDER'} && length($ENV{'NSORDER'}))
        {
            if ($ENV{'NSORDER'} =~ m/^local/) { $cache = 1; }
            else { $cache = 0; }
        }
        else
        {    #On AIX -- look at /etc/netsvc.conf
            my $file = "/etc/netsvc.conf";
            if (-e "/etc/irs.conf" && !-e "/etc/netsvc.conf")
            {
                $file = "/etc/irs.conf";
            }
            if (open(CONF, $file))
            {
                my $line;
                while ($line = <CONF>)
                {
                    chomp $line;
                    if ($line !~ m/^\#/ && $line =~ m/^\s*hosts/)
                    {    #if it doesn't start with a commnet
                        if ($line =~ m/^\s*hosts\s*=\s*local/)
                        {
                            $cache = 1;
                        }
                        last;
                    }
                }
                close(CONF);
            }
        }

    }
    if ($cache == 0) { return; }
    my $file = "/etc/hosts";
    if (open(HOSTS, $file))
    {

        #print "CACHING ETC HOSTS\n";
        my $line;
        while ($line = <HOSTS>)
        {
            chomp $line;
            my ($before, $after) = split "#", $line;
            if ($before)
            {
                my ($ip, $longhost, $shorthost) = split /\s+/, $before;

                #print "IP =|$ip|\tlonghost =|$longhost|\tshorthost =|$shorthost|\n";
                if (length($longhost) && length($ip))
                {
                    $NodeUtils::Hosts{'Hostname'}{$longhost}{'Name'} =
                      $longhost;
                    $NodeUtils::Hosts{'Hostname'}{$longhost}{'IP'} = $ip;
                    $NodeUtils::Hosts{'IP'}{$ip} = $longhost;
                    if (length($shorthost))
                    {
                        $NodeUtils::Hosts{'Hostname'}{$shorthost}{'Name'} =
                          $longhost;
                        $NodeUtils::Hosts{'Hostname'}{$shorthost}{'IP'} = $ip;
                    }
                }

            }

        }
        close(HOSTS);
    }

}

#--------------------------------------------------------------------------------

=head3    determineMS

        Get the name of the localhost from an array of node names.

        Arguments:
        reference to a list of nodeNames
        Returns:
                Name of the local host
        Globals:
                $::PLTFRM
        Error:
                undefined
        Example:
        :$::MS_NODE = InstallUtils->determineMS(\@::NODELIST);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub determineMS
{
    my ($class, $refnodes) = @_;
    my @nodes = @$refnodes;
    my ($localnode, $ifconfig, $label, $separator, $address, @ipaddrs,
        %localhostnames);

    if ($::PLTFRM eq "AIX")
    {
        $ifconfig = "/usr/sbin/ifconfig -a";

        # format is: "inet xxx.xxx.xxx.xxx
        $label     = "inet";
        $separator = " ";
    }
    else
    {    #Linux
        $ifconfig = "/sbin/ifconfig -a";

        # format is: "inet addr:xxx.xxx.xxx.xxx
        $label     = "inet addr";
        $separator = ":";
    }

    NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils', 'V',
                              'IMsgCMD', $ifconfig);
    my @ips = `$ifconfig 2>&1`;
    my $rc  = $? >> 8;
    if ($rc)
    {
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'E', 'EMsgCANT_RUN', $ifconfig, $rc);
    }
    foreach (@ips)
    {
        if ($_ =~ /$label$separator((\d{1,3}?\.){3}(\d){1,3})\s/o)
        {

            #print "found \"$1\" in \"$_\"\n";
            push @ipaddrs, $1;
        }
    }
    foreach my $addr (@ipaddrs)
    {

        #see what it resolves to
        $localhostnames{NodeUtils->tryHost($addr)} = 1;
    }
  FINDMS: foreach my $name (@nodes)
    {
        my $rn = NodeUtils->tryHost($name);
        if ($localhostnames{$rn})
        {

            #there should only be one ManagedNode that maps to the localhost
            $localnode = $name;
            last FINDMS;
        }
    }
    return $localnode;
}

#--------------------------------------------------------------------------------

=head3    export_csminstall

        NFS Export the /csminstall directory on the Management Server

        Arguments:
                $dir_to_be_exported
        Returns:
                $::OK
                $::NOK
        Globals:
                $::PLTFRM 
                %::MS_INSTALL_SERVER
                %::INSTALL_SERVERS
        Error:
                $::NOK
        Example:
                if(NetworkUtils->export_csminstall('/csminstall/csm') != 0) {
                    blah;
                }
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub export_csminstall
{
    my ($class, $dir_to_be_exported) = @_;
    my $cmd;
    my $rc = 0;

    #my $PLTFRM = `uname`; chomp $PLTFRM;    #using $::PLTFRM instead
    if ($::MS_INSTALL_SERVER)
    {    #the MS is serving NFS files
        if (-d $dir_to_be_exported)
        {
            if ($::PLTFRM eq "AIX")
            {

                #$cmd = "$::EXPORTFS -i -o ro /csminstall";
                $cmd = "$::EXPORTFS -i -o ro $dir_to_be_exported";
                NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                          'NodeUtils', 'V', 'IMsgCMD', $cmd);
                $rc = system($cmd) >> 8;
                if ($rc)
                {
                    NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
                }

            }
            else
            {

                #$cmd = "$::EXPORTFS -o ro,no_root_squash \*:/csminstall";
                $cmd =
                  "$::EXPORTFS -o sync,ro,no_root_squash \*:$dir_to_be_exported";
                NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                          'NodeUtils', 'V', 'IMsgCMD', $cmd);
                $rc = system($cmd) >> 8;
                if ($rc)
                {
                    NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'E', 'EMsgCommandFailed', $cmd, $rc);
                }
            }
        }
        else
        {
            NodeUtils->messageFromCat(
                                      'csmInstall.cat',
                                      $::MSGMAPPATH,
                                      'csminstall',
                                      'W',
                                      'EMsgNo_Dir_Export',
                                      $dir_to_be_exported
                                      );
            $rc = 1;
        }
    }

    #determine AIX and Linux install servers
    my %AIX;
    my %Linux;
    foreach my $key (keys %::INSTALL_SERVERS)
    {
        foreach my $server (keys %{$::INSTALL_SERVERS{$key}})
        {
            my $os  = $::INSTALL_SERVERS{$key}{$server}{'OS'};
            my $dir = $::INSTALL_SERVERS{$key}{$server}{'dir'};
            if ($os eq "AIX")
            {
                $AIX{$dir}{$server} = 1;
            }
            elsif ($os eq "Linux")
            {
                $Linux{$dir}{$server} = 1;
            }
            else
            {
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E1', 'EMsgINVALID_OSTYPE');
            }
        }
    }
    foreach my $dir (keys %AIX)
    {    #there are AIX install servers, this code will be done in rigel
        my $list     = join ",", keys %{$AIX{$dir}};
        my $real_dir = $dir_to_be_exported;

        #check to see if $dir is already exported
        $cmd = "$::DSH -n $list \"$::EXPORTFS | $::GREP $dir\" 2>&1";
        NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                  'V', 'IMsgCMD', $cmd);
        my @output = `$cmd`;
        my @unexport;
        foreach my $line (@output)
        {
            chomp $line;
            my ($hostname, $line) = split ':', $line;
            if ($line =~ m/$dir/)
            {
                push @unexport, $hostname;
            }
        }
        $real_dir =~ s/^\/csminstall/$dir/;
        if (@unexport)
        {
            $cmd = "$::DSH -n $list '$::EXPORTFS -i -u $real_dir'";
            NodeUtils->runcmd($cmd, 0);
        }
        $cmd = "$::DSH -n $list '$::EXPORTFS -i -o ro $real_dir\'";
        NodeUtils->runcmd($cmd, 0);
    }
    foreach my $dir (keys %Linux)
    {
        my $list     = join ",", keys %{$Linux{$dir}};
        my $real_dir = $dir_to_be_exported;

        #check to see if $dir is already exported
        $cmd = "$::DSH -n $list \"$::EXPORTFS | $::GREP $dir\" 2>&1";
        NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                  'V', 'IMsgCMD', $cmd);
        my @output = `$cmd`;
        my @unexport;
        foreach my $line (@output)
        {
            chomp $line;
            my ($hostname, $line) = split ':', $line;
            if ($line =~ m/$dir/)
            {
                push @unexport, $hostname;
            }
        }

        $real_dir =~ s/^\/csminstall/$dir/;

        if (@unexport)
        {
            $cmd = "$::DSH -n $list '$::EXPORTFS -u \*:$real_dir'";
            NodeUtils->runcmd($cmd, 0);
        }

        $cmd =
          "$::DSH -n $list \"$::EXPORTFS -o sync,ro,no_root_squash \*:$real_dir\"";
        NodeUtils->runcmd($cmd, 0);
    }
    return ($rc);
}

#--------------------------------------------------------------------------------

=head3    getHost

    Return primary hostname and ip address for the given hostname or
    ip address and die if hostname resolution fails.

    Arguments:
        A string: either an Ip Address or a HostName.
    Returns:
         ($nodename, $nodeip) as strings
    Globals:
        $NodeUtils::errno
        $NodeUtils::Hosts
    Error:
        returns 0;
        sets NodeUtils::errno.
    Example:
        my ($nodename, $nodeip) = NetworkUtils->getHost($node);
    Comments:
        none

=cut

#--------------------------------------------------------------------------------

sub getHost
{
    my ($class, $arg, $live) = @_;

    #print "arg=$arg\n";
    my ($hostname, $ipaddr);
    if (!$NodeUtils::gethost{$arg})
    {    #not cached
        if (NetworkUtils->isIpaddr($arg))
        {
            if (defined $NodeUtils::Hosts{'IP'}{$arg})
            {

                #cached from /etc/hosts
                $hostname = $NodeUtils::Hosts{'IP'}{$arg};

                #print "using cached IP: $arg\n";
            }
            else
            {
                $ipaddr = $arg;
                my $packedaddr = inet_aton($ipaddr);
                $hostname = gethostbyaddr($packedaddr, AF_INET);
                if (!length($hostname) && $live)
                {
                    MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET,
                                              "E27", 'EMsgBAD_IPADDR', $ipaddr);
                }
                if (!length($hostname))
                {
                    MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET,
                                              "E27", 'EMsgBAD_IPADDR', $ipaddr);
                }

                #print "hostname=$hostname\n";
            }
        }
        else    # they specified a hostname
        {
            if (defined $NodeUtils::Hosts{'Hostname'}{$arg})
            {
                $hostname = $NodeUtils::Hosts{'Hostname'}{$arg}{'Name'};
                $ipaddr   = $NodeUtils::Hosts{'Hostname'}{$arg}{'IP'};

                #print "using cached Hostname: $arg\n";
            }
            else
            {
                $hostname = $arg;    # this may be a short hostname
                my ($name, $aliases, $addrtype, $length, @addrs) =
                  gethostbyname($hostname);
                if (!length($name) && $live)
                {
                    MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET,
                                          "E28", 'EMsgBAD_HOSTNAME', $hostname);
                }
                if (!length($name))
                {
                    MessageUtils->messageFromCat($MSGCAT, $MSGMAPPATH, $MSGSET,
                                          "E28", 'EMsgBAD_HOSTNAME', $hostname);
                }

                #print "name=$name, # of addrs=$#addrs.\n";
                my $packedaddr = $addrs[0];
                $ipaddr   = inet_ntoa($packedaddr);
                $hostname =
                  $name; # they may have specified a shorter or non-primary name
                         #print "ipaddr=$ipaddr, hostname=$hostname\n";
            }
        }
        $NodeUtils::gethost{$arg}{'Name'} = $hostname;    #cache this for later
        $NodeUtils::gethost{$arg}{'IP'}   = $ipaddr;      #cache this for later
    }
    else
    {                                                     #cached
        $hostname = $NodeUtils::gethost{$arg}{'Name'};
        $ipaddr   = $NodeUtils::gethost{$arg}{'IP'};
    }
    return ($hostname, $ipaddr);
}

#--------------------------------------------------------------------------------

=head3    getHost_rveg

	NOTE:  This is a last minute correction for the rveg release,
	which will be distributed to CSM as a whole during rbac.

    Description:

	Returns primary hostname and ip address for a node.
	Either an IP address or a Hostname will be accepted as
	the single parameter.

	Uses a runcmd style error code and message string to return status.

	Status should be checked before accessing the return values, 
	which will both be undef if the command failed.


    Arguments:
        One scalar parameter:  either an Ip Address or a HostName.

    Returns:
	($host_name, $ip_addr)	- the hostname and ip address for parameter

	$::GETHOST_RC		- $::OK on success - this is zero
	$::GETHOST_MSG		- could contain a warning message on success
                                  but currently doesn't.

    Globals:
	$::GETHOST_RC
	@::GETHOST_MSG

    Error:
        Sets $::GETHOST_RC to:

		"E27" bad ip address
		"E28" bad host_name

		or $::OK  on success - this is zero

	Sets @::GETHOST_MSG to one of:

		'EMsgBAD_IPADDR'
		'EMsgBAD_HOSTNAME'
	
    Example:

        my $node = $someting; # either ip addres or hostname

        my ($host_name, $ip_address) = NetworkUtils->getHost($node);

	#  if you're not concerned with status and just want to know
	#  if things worked out, use something like this:

        # retun values are undef on error
        if ( $::GETHOST_RC != $::OK )
	{
	    next;
	}
 	# otherwise the $nodename and $nodeip are good.

    Comments:
        none

=cut

#--------------------------------------------------------------------------------

sub getHost_rveg
{
    my ($class, $arg) = @_;
    my ($hostname, $ipaddr, $packedaddr);

    # glob status
    $::GETHOST_RC  = $::OK;    # reassigned on error
    $::GETHOST_MSG = undef;    # reassigned on error

    if ($NodeUtils::gethost{$arg})
    {

        # print "returning cached values \n";
        $hostname = $NodeUtils::gethost{$arg}{'Name'};
        $ipaddr   = $NodeUtils::gethost{$arg}{'IP'};

        return ($hostname, $ipaddr);
    }

    # do the IP address parameter

    if (NetworkUtils->isIpaddr($arg))
    {
        $ipaddr = $arg;

        if (defined $NodeUtils::Hosts{'IP'}{$arg})
        {

            # print "returning cached IP: $ipaddr\n";
            $hostname = $NodeUtils::Hosts{'IP'}{$ipaddr};

            return ($hostname, $ipaddr);

        }

        # IP not cached
        $packedaddr = inet_aton($ipaddr);
        $hostname   = gethostbyaddr($packedaddr, AF_INET);

        if (!$hostname)    # return on error
        {
            $::GETHOST_RC  = 27;
            $::GETHOST_MSG = 'EMsgBAD_IPADDR';

            return (undef, undef);
        }

        # cache the  values
        $NodeUtils::gethost{$arg}{'Name'} = $hostname;
        $NodeUtils::gethost{$arg}{'IP'}   = $ipaddr;

        return ($hostname, $ipaddr);
    }

    # do the hostname parameter

    $hostname = $arg;

    if (defined $NodeUtils::Hosts{'Hostname'}{$hostname})
    {

        # print "using cached Hostname: $hostname\n";
        return ($hostname, $ipaddr);
    }

    # high overhead call...
    my ($name, $aliases, $addrtype, $length, @addrs) = gethostbyname($hostname);

    if (!$name)    # return on error
    {

        $::GETHOST_RC  = 28;
        $::GETHOST_MSG = 'EMsgBAD_HOSTNAME';
        return (undef, undef);
    }

    # print "found name=$name, and addrs=$#addrs.\n";

    $packedaddr = $addrs[0];
    $ipaddr     = inet_ntoa($packedaddr);
    $hostname   = $name;                    # may be a short name

    # cache the  values
    $NodeUtils::gethost{$hostname}{'Name'} = $hostname;
    $NodeUtils::gethost{$hostname}{'IP'}   = $ipaddr;

    return ($hostname, $ipaddr);

}

#--------------------------------------------------------------------------------

=head3    get_management_server

        Get the hostname of the management server.

        Arguments:
                $node_hostname
        Returns:
                ($hostname, $ipaddr) or $hostname, depending on the value of
                wantarray (?)
        Globals:
                $::GETSOURCEIP2TARGET
        Error:
                undefined
        Example:
                ($ms_hostname, $ipaddr) = NetworkUtils->get_management_server_by_hostname;
        Comments:
                Since the nodes might need to contact the management server through a
                different address than its primary hostname, we cannot just use the output
                of the hostname command here.  Use the getSourceIP2Target command instead.

                If we cannot determine the route to the node via getSourceIP2Target
                just return nothing.  This indicates that there is no way to get to the node.

                Note:  caller is reponsible for verifying return values.

=cut

#--------------------------------------------------------------------------------

sub get_management_server
{
    my ($class, $node_hostname) = @_;
    my $ipaddr   = "";
    my $hostname = "";

    # Move this to InstallDefs.pm.perl
    $::GETSOURCEIP2TARGET = "/opt/csm/csmbin/getSourceIP2Target";

    my $cmd    = "$::GETSOURCEIP2TARGET $node_hostname";
    my @output = NodeUtils->runcmd($cmd, 0);
    my $rc     = $::RUNCMD_RC;
    if (!$rc)    # If no errors
    {
        $ipaddr = $output[0];

        if ($ipaddr)
        {
            $hostname = NodeUtils->tryHost($ipaddr);
        }
    }

    if (wantarray) { return ($hostname, $ipaddr); }
    else { return $hostname; }
}

#--------------------------------------------------------------------------------

=head3    getRSCThostname

        This routine is used to always return the same hostname regardless 
        of hostname resolution settings. It returns the RSCT node id of the 
        current machine.

        Since the RSCT node id is unchangable by the user, the only way
        this could change is if RSCT is uninstalled, and reinstalled.
 

    Arguments:
        none
    Returns:
        $ref from lsrsrc-api call.
    Globals:
        none
    Error:
        none
    Example:
        none
    Comments:
        none

=cut

#--------------------------------------------------------------------------------

sub getRSCThostname
{

    #my $outref =   NodeUtils->runcmd('CT_MANAGEMENT_SCOPE=1 /usr/bin/lsrsrc-api -s IBM.Host::::NodeNameList');
    #$outref =~ s/{|}//g;
    #now using the RSCT node id because the NodeNameList changes when the output of hostname changes
    my $outref = NodeUtils->runcmd("/usr/sbin/rsct/bin/lsnodeid");
    return $outref;
}

#--------------------------------------------------------------------------------

=head3    groupNFSnodes

        I wonder ...

        Arguments:
        \%DestNodeHash    - InstallServer must be an attr of each node in hash.
          Fanout value
        \@nodeArray    - If caller is the only dsh to a select group
        Returns:
                I wonder...
        Globals:
                none
        Error:
                none
        Example:
         my $fan = NetworkUtils->groupNFSnodes( \%nodeHash, $csmFanout );
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub groupNFSnodes
{
    my ($class, $DestNodeHash, $fanout, $nodes) = @_;

    #put nodes into hashes based on install server
    my %is;
    my @norm;
    my @slist;
    if ($nodes)
    {
        foreach my $node (@$nodes)
        {
            my $server = $$DestNodeHash{$node}{'InstallServer'};
            if ($server)
            {    #there is an install server for this node
                    #$is{$server}{$node}=1;
                if (!exists $is{$server})
                {    #haven't scene this one before
                    push @slist, $server;
                }
                push @{$is{$server}}, $node;
            }
            else
            {        #just uses the ms as an install server
                     #$norm{$node}=1;
                push @norm, $node;

                #print "pushed node $node onto norm\n";
            }
        }
    }
    else
    {                #using the entire hash of nodes
        foreach my $node (keys %$DestNodeHash)
        {
            my $server = $$DestNodeHash{$node}{'InstallServer'};
            if ($server)
            {        #there is an install server for this node
                     #$is{$server}{$node}=1;
                if (!exists $is{$server})
                {    #haven't scene this one before
                    push @slist, $server;
                }
                push @{$is{$server}}, $node;
            }
            else
            {        #just uses the ms as an install server
                     #$norm{$node}=1;
                push @norm, $node;

                #print "pushed node $node onto norm\n";
            }
        }
    }
    my %dsh;
    my $num_servers = (scalar @slist) + 1;
    my $num_nodes;
    if ($nodes) { $num_nodes = scalar @$nodes; }
    else { $num_nodes = scalar keys %$DestNodeHash; }

    #print "the number of install servers is $num_servers -1\nthe number of nodes is $num_nodes\nthe fanout is $fanout\n";
    my $count    = 0;
    my $group    = 0;
    my $fancount = 0;
  GROUPNODES: while ($num_nodes)
    {

        #print "in GROUPNODES list: count=$count\tgroup=$group\tnum_nodes=$num_nodes\n";
        if ($fancount == $fanout)
        {    #we are done with one list
            $fancount = 0;
            $group++;
        }
        if ($count > $num_servers)
        {
            $count = 0;
        }
        if ($count == $num_servers)
        {
            $count = 0;
        }
        if ($count == 0)
        {    #grab one from the normal list if any are left
            if (@norm)
            {

                #print "pushing onto group $group\n";
                push @{$dsh{$group}}, shift @norm;
                $num_nodes--;
                $fancount++;
                $count++;
                next GROUPNODES;
            }
        }
        while (!@{$is{$slist[$count - 1]}})
        {    #none in this array
                #print "server $slist[$count] is not defined\n";
                #print "incrementing count $count\n";
            $count++;
            if ($count == $num_servers || !($slist[$count - 1]))
            {
                next GROUPNODES;
            }
        }
        push @{$dsh{$group}}, shift @{$is{$slist[$count - 1]}};
        $fancount++;
        $num_nodes--;
        $count++;
    }
    my %fan = ();
    foreach my $g (keys %dsh)
    {
        @{$fan{$g}{'ALL'}} = @{$dsh{$g}};
        foreach my $node (@{$dsh{$g}})
        {
            my $server = $$DestNodeHash{$node}{'InstallServer'};
            my ($s_hostname, $dir) = split(/:/, $server);
            if (!$server)
            {
                $server = $$DestNodeHash{$node}{'ManagementServer'};
            }
            else
            {    #server exists
                if ($server !~ m/:/)
                {    #doesn't have a directory
                    $server .= ":$::SERVER_DIRECTORY";
                }
                if ($$DestNodeHash{$node}{'ManagementServer'} eq
                    $$DestNodeHash{$node}{'Hostname'})
                {

                    #this node is also the MS
                    $server = $$DestNodeHash{$node}{'ManagementServer'};
                }
                elsif ($s_hostname eq $$DestNodeHash{$node}{'ManagementServer'})
                {
                    $server = $$DestNodeHash{$node}{'ManagementServer'};
                }
            }
            push @{$fan{$g}{'Server'}{$server}}, $node;
        }
        my @s = keys %{$fan{$g}{'Server'}};

        #print "group $g has nodes @{$dsh{$g}}\tand servers @s \n";
    }
    return (\%fan);
}

#--------------------------------------------------------------------------------

=head3    nfsprobe

         Probe NFS status like the "nfs probe" command on redhat

        Arguments:
                none
        Returns:
                NFS service command (e.g. start, restart or reload).
        Globals:
                none
        Error:
                retCode from NodeUils->service("nfs","probe");
        Example:
                $serviceCmd = NetworkUtils->nfsprobe();

        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub nfsprobe
{
    my $CHECKPROC = "/sbin/checkproc";
    my $PIDOF     = "/sbin/pidof";
    my $NFSD      = "nfsd";
    my $MOUNTD    = "/usr/sbin/rpc.mountd";
    my $EXPORTS   = "/etc/exports";
    my $ETAB      = "/var/lib/nfs/etab";
    my ($nfsd_status, $mountd_status);
    my $cmd = NodeUtils->service("nfs", "probe");

    #my $output=NodeUtils->runcmd($cmd,0);
    my $output = `$cmd 2>&1`;
    if (!$?)
    {
        return $output;
    }

    NodeUtils->runcmd("$CHECKPROC -n $NFSD", -1);
    $nfsd_status = $::RUNCMD_RC;

    NodeUtils->runcmd("$CHECKPROC $MOUNTD", -1);
    $mountd_status = $::RUNCMD_RC;

    #either nfsd and mountd are not start
    if ($nfsd_status || $mountd_status)
    {
        return "start";
    }

    NodeUtils->runcmd("$PIDOF $NFSD", -1);
    $nfsd_status = $::RUNCMD_RC;

    NodeUtils->runcmd("$PIDOF $MOUNTD", -1);
    $mountd_status = $::RUNCMD_RC;

    if ($nfsd_status == 1 || $mountd_status == 1)
    {
        return "restart";
    }

    my $exports_mtime = (stat($EXPORTS))[9];
    my $etab_mtime    = (stat($ETAB))[9];

    if ($exports_mtime > $etab_mtime)
    {
        return "reload";
    }

    return "status";
}

#--------------------------------------------------------------------------------

=head3    service

    Send a service request to an init script.

    Arguments:
        $service  - a service name such as 'inetd','xinetd'
        $svcarg   - arguments for the service such as 'start',
                    'stop' or 'status'. 
    Returns:
        A full cli for the service script.
    Globals:
        none
    Error:
        none
    Example:
        none
    Comments:
        none

=cut

#--------------------------------------------------------------------------------

sub service
{
    my ($class, $service, $svcarg) = @_;

    my $cmd;
    my $SVCCLI = "/sbin/service";
    my $SVCDIR = "/etc/init.d";
    my $distro = NodeUtils->distribution();

    #  On SuSE, nfs server script is "nfsserver".
    if ($distro =~ /(SuSE)|(SLES)/ && $service eq "nfs")
    {
        $service = "nfsserver";
    }

    if (-f $SVCCLI)
    {
        $cmd = "$SVCCLI $service $svcarg ";
    }
    else
    {
        $cmd = "$SVCDIR/$service $svcarg";
    }

    return $cmd;
}

#--------------------------------------------------------------------------------

=head3    start_nfs

        Starts NFS services on the managment server

        Arguments:
                none
        Returns:
                $::OK
                $::NOK
        Globals:
                $::PLTFRM 
                $::SERVICE
                %::INSTALL_SERVERS
        Error:
                $::NOK
        Example:
                if (NetworkUtils->start_nfs != 0) { blah; }_
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub start_nfs
{
    my ($cmd, $output);
    my $rc      = 0;
    my $exports = "/etc/exports";

    #my $PLTFRM = `uname`; chomp $PLTFRM;      #using $::PLTFRM instead
    if ($::MS_INSTALL_SERVER)
    {    #the MS is serving NFS files
            # Ensure nfs is running.
        if (!-s $exports)
        {

            # nfs will only start if /etc/exports has a size > 0
            open(EXPORTS, ">$exports")
              || NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'E', 'EMsgCANT_READ_FILE', $exports);
            print EXPORTS "\n";
            close(EXPORTS);
        }

        # Start nfs if needed.
        if ($::PLTFRM eq "AIX")
        {
            my $cmd = "$::MKNFS -B  >/dev/null 2>&1";
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cmd);
            my $rc = system($cmd) >> 8;
            if ($rc)
            {
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }

        }
        else
        {

            # We need modify start_nfs in rc.config on SuSE
            my $distro = NodeUtils->distribution();
            if ($distro =~ /SuSE/ || $distro =~ /SLES/)
            {
                my $RCCONFIG = "/etc/rc.config";
                my $s        = NodeUtils->readFile($RCCONFIG);
                $s =~
                  s/^\s*NFS_SERVER\s*=\s*"\s*[Nn][Oo]\s*"\s*$/NFS_SERVER="yes"/m;
                if (open(OUT, ">$RCCONFIG"))
                {
                    print OUT ($s);
                    close(OUT);
                }
                else
                {
                    NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                           'csminstall', 'I', 'IMsgShow_Output',
                                           'Can not open $RCCONFIG for write');
                }
            }

            # The output of the 'service nfs probe' tells
            #    what needs to be done to start nfs.  It returns either
            #    'start', 'restart', 'reload' or nothing.

            # SuSE Change
            #$output = `$::SERVICE nfs probe 2>&1`;
            $output = InstallUtils->nfsprobe();

            if ($output)
            {

                # SuSE Change
                #$cmd = "$::SERVICE nfs $output";
                $cmd = NodeUtils->service("nfs", $output);

                NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                          'NodeUtils', 'V', 'IMsgCMD', $cmd);
                $rc = system($cmd) >> 8;
                if ($rc)
                {
                    NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'E', 'EMsgCommandFailed', $cmd, $rc);
                }
                else
                {

                    #wait for nfsserver to start on SLES7
                    sleep(1);
                }
            }
        }
    }
    if (defined %::INSTALL_SERVERS)
    {    #There are install servers serving files
            #get info about install servers
        my @make_ex        = ();
        my @AIX_servers    = ();
        my @RedHat_servers = ();
        my @SUSE_servers   = ();

        #my @Linux_servers = ();
        my %service;
        if (defined %{$::INSTALL_SERVERS{'isnode'}})
        {

            #if some of the install servers are nodes then this is a lot easier
            my @servers = keys %{$::INSTALL_SERVERS{'isnode'}};
            foreach my $s (@servers)
            {
                my $os     = $::INSTALL_SERVERS{'isnode'}{$s}{'OS'};
                my $distro = $::INSTALL_SERVERS{'isnode'}{$s}{'DISTRO'};
                if ($os eq "AIX")
                {
                    push @AIX_servers, $s;
                }
                elsif ($os eq "Linux")
                {
                    if ($distro =~ m/RedHat/)
                    {
                        push @RedHat_servers, $s;
                    }
                    elsif ($distro =~ m/SuSE/ || $distro =~ m/SLES/)
                    {    #suse?
                        push @SUSE_servers, $s;
                    }

                    #    push @Linux_servers, $s;
                }
            }
            if (@RedHat_servers)
            {
                my $list = join ',', @RedHat_servers;
                my $cmd =
                  "LANG=C $::DSH -n $list \"$::LS -s $exports; /sbin/service nfs probe;\"";
                NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                          'NodeUtils', 'V', 'IMsgCMD', $cmd);
                my @output = `$cmd`;
                foreach my $line (@output)
                {
                    chomp $line;
                    my ($server, $data) = split ":", $line, 2;
                    if ($line =~ m/$exports/)
                    {    #this is the ls line
                        if (   $line !~ m/"does not exist"/
                            || $line !~ m/"not found"/)
                        {

                            #then /etc/exports exists
                            $data =~ s/^\s+//;    #remove any leading whitespace
                            my ($size, $file) = split /\s+/, $data, 2;
                            if ($size == 0)
                            {
                                push @make_ex, $server;
                            }
                        }
                        else
                        {                         #/ext/exports does not exist
                            push @make_ex, $server;
                        }
                    }
                    elsif (   ($data =~ m/start/)
                           || ($data =~ m/restart/)
                           || ($data =~ m/reload/))
                    {
                        $data =~ s/^\s+//;        #remove any leading whitespace
                        push @{$service{$data}}, $server;
                    }
                }
            }
            if (@AIX_servers || @SUSE_servers)
            {
                my $list = join ',', @SUSE_servers, @AIX_servers;

                #make sure /exports is larger than 0
                my $cmd = "LANG=C $::DSH -n $list \"$::LS -s $exports;\" 2>&1";
                NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                          'NodeUtils', 'V', 'IMsgCMD', $cmd);
                my @output = `$cmd`;
                foreach my $line (@output)
                {
                    chomp $line;
                    my ($server, $data) = split ":", $line, 2;
                    if ($line =~ m/$exports/)
                    {    #this is the ls line
                        if (   $line !~ m/"does not exist"/
                            || $line !~ m/"not found"/)
                        {

                            #then /etc/exports exists
                            $data =~ s/^\s+//;    #remove any leading whitespace
                            my ($size, $file) = split /\s+/, $data, 2;
                            if ($size == 0)
                            {
                                push @make_ex, $server;
                            }
                        }
                        else
                        {                         #no file
                            push @make_ex, $server;
                        }
                    }
                }
            }
        }
        if (defined %{$::INSTALL_SERVERS{'notnode'}})
        {

            #size of /etc/exports
            #OS
            my $dsh_list = join ",", (keys %{$::INSTALL_SERVERS{'notnode'}});
            my $cmd =
              "LANG=C $::DSH -n $dsh_list \"$::LS -s $exports; /bin/uname; /bin/cat /etc/*release; /sbin/service nfs probe;\" 2>&1";
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cmd);
            my @output = `$cmd`;
            foreach my $line (@output)
            {
                chomp $line;
                my ($server, $data) = split ":", $line, 2;
                if ($line =~ m/$exports/)
                {    #this is the ls line
                    if ($line !~ m/"does not exist"/ || $line !~ m/"not found"/)
                    {

                        #then /etc/exports exists
                        $data =~ s/^\s+//;    #remove any leading whitespace
                        my ($size, $file) = split /\s+/, $data, 2;
                        if ($size == 0)
                        {
                            push @make_ex, $server;
                        }
                    }
                    else
                    {
                        push @make_ex, $server;
                    }
                }
                elsif ($data =~ m/AIX/)
                {
                    $::INSTALL_SERVERS{'notnode'}{$server}{'OS'} = "AIX";
                    push @AIX_servers, $server;
                }
                elsif ($data =~ m/Red Hat/)
                {
                    $::INSTALL_SERVERS{'notnode'}{$server}{'OS'}     = "Linux";
                    $::INSTALL_SERVERS{'notnode'}{$server}{'DISTRO'} = "RedHat";
                    push @RedHat_servers, $server;
                }
                elsif ($data =~ m/SuSE/i)
                {
                    $::INSTALL_SERVERS{'notnode'}{$server}{'OS'}     = "Linux";
                    $::INSTALL_SERVERS{'notnode'}{$server}{'DISTRO'} = "SUSE";
                    push @SUSE_servers, $server;
                }
                elsif (   ($data =~ m/start/)
                       || ($data =~ m/restart/)
                       || ($data =~ m/reload/))
                {
                    $data =~ s/^\s+//;    #remove any leading whitespace
                    push @{$service{$data}}, $server;
                }
            }
        }

        #first make sure /etc/exports has a size > 0 on all of the install servers
        if (@make_ex)
        {
            my $ex_list = join ",", @make_ex;
            $cmd = "$::DSH -n $ex_list \"/bin/echo ' ' >> $exports\" 2>&1";
            NodeUtils->runcmd($cmd, 0);
        }
        if (@RedHat_servers)
        {

            #/sbin/service
            #output of service nfs probe
            #then start NFS
            #special steps required in SuSE / SLES case -- these will be implemented in rigel
            foreach my $value (keys %service)
            {
                my $list = join ",", @{$service{$value}};
                $cmd = "$::DSH -n $list \"/sbin/service nfs $value &\" 2>&1";
                NodeUtils->runcmd($cmd, 0);
            }
        }
        if (@SUSE_servers)
        {
            my $list = join ',', @SUSE_servers;

            #going to need to run sed command to edit /etc/rc.config on SUSE server
            my $RCCONFIG  = "/etc/rc.config";
            my $CHECKPROC = "/sbin/checkproc";
            my $PIDOF     = "/sbin/pidof";
            my $NFSD      = "nfsd";
            my $MOUNTD    = "/usr/sbin/rpc.mountd";
            my $EXPORTS   = "/etc/exports";
            my $ETAB      = "/var/lib/nfs/etab";

            #$cmd = qq($::DSH -n $list '/bin/sed -e s/.*NFS_SERVER.*/NFS_SERVER=\\"yes\\"/ $RCCONFIG | cat '-' > /tmp/csm.rcconfig; /bin/mv /tmp/csm.rcconfig $RCCONFIG; $CHECKPROC -n $NFSD; echo \"nfsd proc rc=\$?\"; $CHECKPROC $MOUNTD; /bin/echo \"mountd proc rc=\$?\"; $PIDOF $NFSD; echo \"pidof nfsd rc=\$?\"; $PIDOF $MOUNTD; echo \"pidof mountd rc=\$?\"; /usr/bin/stat -c %n=%Y $EXPORTS; /usr/bin/stat -c %n=%Y $ETAB;');
            $cmd =
              qq($::DSH -n $list '/bin/sed -e s/.*NFS_SERVER.*/NFS_SERVER=\\"yes\\"/ $RCCONFIG | cat '-' > /tmp/csm.rcconfig; /bin/mv /tmp/csm.rcconfig $RCCONFIG; $CHECKPROC -n $NFSD; echo \"nfsd proc rc=\$?\"; $CHECKPROC $MOUNTD; /bin/echo \"mountd proc rc=\$?\"; $PIDOF $NFSD; echo \"pidof nfsd rc=\$?\"; $PIDOF $MOUNTD; echo \"pidof mountd rc=\$?\";');
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cmd);
            my @output = `$cmd`;
            my %service_hash;
            my %suse_service = ();
            foreach my $line (@output)
            {

                # print "LINE = $line";
                chomp $line;
                my $hostname;
                ($hostname, $line) = split ':', $line, 2;

                # print "HOSTNAME = $hostname\n";
                my $rc = (split '=', $line)[1];
                if ($line =~ m/nfsd proc/)
                {
                    $service_hash{$hostname}{'NFSD'}{'PROC'} = $rc;

                    #    print "NFSD PROC = $rc\n";
                }
                elsif ($line =~ m/mountd proc/)
                {
                    $service_hash{$hostname}{'MOUNTD'}{'PROC'} = $rc;

                    # print "MOUNTD PROC = $rc\n";
                }
                elsif ($line =~ m/pidof nfsd/)
                {
                    $service_hash{$hostname}{'NFSD'}{'PIDOF'} = $rc;

                    #    print "NFSD PIDOF = $rc\n";
                }
                elsif ($line =~ m/pidof mountd/)
                {
                    $service_hash{$hostname}{'MOUNTD'}{'PIDOF'} = $rc;

                    #    print "MOUNTD PIDOF = $rc\n";
                }
                elsif ($line =~ m/$EXPORTS/)
                {
                    $service_hash{$hostname}{'EXPORTS'} = $rc;
                }
                elsif ($line =~ m/$ETAB/)
                {
                    $service_hash{$hostname}{'ETAB'} = $rc;
                }
            }
            foreach my $hostname (keys %service_hash)
            {
                my $value;
                if (   $service_hash{$hostname}{'NFSD'}{'PROC'}
                    || $service_hash{$hostname}{'MOUNTD'}{'PROC'})
                {
                    $value = "start";
                }
                elsif (   $service_hash{$hostname}{'NFSD'}{'PIDOF'} == 1
                       || $service_hash{$hostname}{'MOUNTD'}{'PIDOF'} == 1)
                {
                    $value = "restart";
                }
                elsif ($service_hash{$hostname}{'EXPORTS'} >
                       $service_hash{$hostname}{'ETAB'})
                {
                    $value = "reload";
                }
                else
                {
                    $value = "status";
                }

                # print "VALUE=$value\n\n";
                push @{$suse_service{$value}}, $hostname;
            }
            foreach my $value (keys %suse_service)
            {
                my $list = join ',', @{$suse_service{$value}};
                my $cmd =
                  "$::DSH -n $list \"PATH=\$PATH:/sbin ; /etc/init.d/nfsserver $value;\"";

                NodeUtils->runcmd($cmd, 0);
            }
        }
        if (@AIX_servers)
        {
            my $list = join ',', @AIX_servers;
            $cmd = "$::DSH -n $list \"$::MKNFS -B >/dev/null\" 2>&1";
            NodeUtils->runcmd($cmd, 0);
        }
    }
    return ($rc);
}

#--------------------------------------------------------------------------------

=head3    tryHost

    Try to return the primary hostname for the given hostname or
    ip address.  If resolution fails, return the argument passed in.
        

    Arguments:
        hostname and/or ip address as a string
    Returns:
        primary hostname as a string or given argument if
        the hostname can's be resolved
 
    Globals:
        $NodeUtils::Hosts
        
    Error:
        undefined
    Example:
         my $hostname =
            NetworkUtils->tryHost($node, \$resolved);
    Comments:
        none

=cut

#--------------------------------------------------------------------------------

sub tryHost
{
    my ($class, $arg, $resolvedref) = @_;
    if (!defined $NodeUtils::tryhost{$arg})
    {
        my $host;
        $resolvedref && ($$resolvedref = 0);
        my $res = 0;    #for caching purposes
        if (NetworkUtils->isIpaddr($arg))
        {
            if (defined $NodeUtils::Hosts{'IP'}{$arg})
            {

                #cached from /etc/hosts
                #print "using cached IP: $arg\n";
                $host = $NodeUtils::Hosts{'IP'}{$arg};
                $resolvedref && ($$resolvedref = 1);
                $res = 1;
            }
            else
            {
                my $packedaddr = inet_aton($arg);
                $host = gethostbyaddr($packedaddr, AF_INET);
                if (!length($host)) { $host = $arg; }
                else { $resolvedref && ($$resolvedref = 1); $res = 1; }
            }
        }
        else    # they specified a hostname, but it may be a short name
        {
            if (defined $NodeUtils::Hosts{'Hostname'}{$arg})
            {

                #cached from /etc/hosts
                #print "using cached hostname: $arg\n";
                $host = $NodeUtils::Hosts{'Hostname'}{$arg}{'Name'};
                $resolvedref && ($$resolvedref = 1);
                $res = 1;
            }
            else
            {
                my ($hostname, $aliases, $addrtype, $length, @addrs) =
                  gethostbyname($arg);
                if (length($hostname))
                {
                    $host = $hostname;
                    $resolvedref && ($$resolvedref = 1);
                    $res = 1;
                }
                else { $host = $arg; }
            }
        }
        $NodeUtils::tryhost{$arg}{'Name'} = $host;    #cache this for later
        if ($resolvedref)
        {
            $NodeUtils::tryhost{$arg}{'resolved'} = $$resolvedref;
        }
        else { $NodeUtils::tryhost{$arg}{'resolved'} = $res; }
        return $host;
    }
    else
    {
        $resolvedref && ($$resolvedref = $NodeUtils::tryhost{$arg}{'resolved'});
        return $NodeUtils::tryhost{$arg}{'Name'};     #use cached version
    }
}

#--------------------------------------------------------------------------------

=head3    unexport_csminstall

        NFS unexport the /csminstall directory on the Management Server

        Arguments:
                $dir_to_be_exported
        Returns:
        0 - All was successful.
        1 - An error occured.
        Globals:
                $::PLTFRM 
                %::MS_INSTALL_SERVER
                %::INSTALL_SERVERS
        Error:
               returns 1
        Example:
                if(NetworkUtils->unexport_csminstall('/csminstall/csm') != 0) {
                    blah;
                }
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub unexport_csminstall
{
    my ($class, $dir_to_be_unexported) = @_;
    my $cmd;
    my $rc = 0;

    #my $PLTFRM = `uname`; chomp $PLTFRM;      #using $::PLTFRM instead
    if ($::MS_INSTALL_SERVER)
    {    #the MS is serving NFS files
        if ($::PLTFRM eq "AIX")
        {

            #$cmd = "$::EXPORTFS -i -u /csminstall";
            $cmd = "$::EXPORTFS -i -u $dir_to_be_unexported";
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cmd);
            $rc = system($cmd) >> 8;
            if ($rc)
            {
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }
        }
        else
        {

            #$cmd = "$::EXPORTFS -u \*:/csminstall";
            $cmd = "$::EXPORTFS -u \*:$dir_to_be_unexported";
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cmd);
            $rc = system($cmd) >> 8;
            if ($rc)
            {
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'E', 'EMsgCommandFailed', $cmd, $rc);
            }
        }
    }

    #determine AIX and Linux install servers
    my %AIX;
    my %Linux;
    foreach my $key (keys %::INSTALL_SERVERS)
    {
        foreach my $server (keys %{$::INSTALL_SERVERS{$key}})
        {
            my $os  = $::INSTALL_SERVERS{$key}{$server}{'OS'};
            my $dir = $::INSTALL_SERVERS{$key}{$server}{'dir'};
            if ($os eq "AIX")
            {
                $AIX{$dir}{$server} = 1;
            }
            elsif ($os eq "Linux")
            {
                $Linux{$dir}{$server} = 1;
            }
            else
            {
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E1', 'EMsgINVALID_OSTYPE');
            }
        }
    }
    foreach my $dir (keys %AIX)
    {    #there are AIX install servers, this code will be done in rigel
        my $list = join ",", keys %{$AIX{$dir}};
        my $real_dir = $dir_to_be_unexported;
        $real_dir =~ s/^\/csminstall/$dir/;
        $cmd = "$::DSH -n $list '$::EXPORTFS -i -u $real_dir'";
        NodeUtils->runcmd($cmd, 0);
    }
    foreach my $dir (keys %Linux)
    {
        my $list = join ",", keys %{$Linux{$dir}};
        my $real_dir = $dir_to_be_unexported;
        $real_dir =~ s/^\/csminstall/$dir/;
        $cmd = "$::DSH -n $list '$::EXPORTFS -u \*:$real_dir'";
        NodeUtils->runcmd($cmd, 0);
    }
    return ($rc);
}

#--------------------------------------------------------------------------------

=head2    Remote Shells - RSH, SSH 

=cut

#--------------------------------------------------------------------------------

=head3    checkHOMEforSSHperm

        Checks attributes of root home directory.

        Arguments:
                none
        Returns:
                1 - if $HOME has the wrong permissions
                0 - if permissions are correct.
        Globals:
                none
        Error:
                1 - wrong permissions on root's  $HOME
        Example:
                $rc = NetworkUtils->checkHOMEforSSHperm();
        Comments:
                If the permessions on $HOME are incorrect, then SSH
                should not be setup.

=cut

#--------------------------------------------------------------------------------

sub checkHOMEforSSHperm
{
    my $home      = InstallUtils->getRootHomeDir();
    my $mode      = (stat($home))[2];
    my $real_mode = sprintf "%04o", $mode & 07777;
    my @perm      = split '', $real_mode;
    if (   ($perm[2] == 2)
        || ($perm[2] == 6)
        || ($perm[2] == 7)
        || ($perm[3] == 2)
        || ($perm[3] == 6)
        || ($perm[3] == 7))
    {
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'E', 'EMsgWrongHomePermissions', $home);
        return 1;
    }
    else
    {
        return 0;
    }
}

#--------------------------------------------------------------------------------

=head3    checkRSReachability

        Checks a list of machines (they do not need to be CSM nodes) to make
        sure they can be reached via dsh.

        Arguments:
                reference to an array of hostnames
        Returns:
                reference to an array of unreachable nodes
        Globals:
                $::REMOTE_SHELL
        Error:
                none
        Example:
                 my $unreachable =
                   NetworkUtils->checkRSReachability(\@install_servers);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub checkRSReachability
{

    my ($class, $machines) = @_;
    $::REMOTE_SHELL = InstallUtils->getRemoteShell();
    my $file =
      InstallUtils->make_node_list_file($machines)
      ;    #put all dest nodes in tmp file
    my $cmd;
    (my $cmd_basename = $::REMOTE_SHELL) =~ s:^.*/::;
    if ($cmd_basename ne "rsh" && (InstallUtils->supportedSSHversion()))
    {      #can do SSH with options
        $cmd = "$::REMOTESHELL_EXPECT -ds $file";
    }
    else
    {
        $cmd = "$::REMOTESHELL_EXPECT -d $file";
    }
    my $value = InstallUtils->supportedSSHversion();
    NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils', 'V',
                              'IMsgCMD', $cmd);
    my @output = `$cmd 2>&1`;
    InstallUtils->close_delete_file($::NODE_LIST_FILE, $file); #delete node file
    my %reachable;
    my @unreachable;
    foreach my $line (@output)
    {
        chomp $line;
        my @parts = split ":", $line;
        if ($parts[1] =~ m/test.success/)
        {    #this node is reachable
            $reachable{$parts[0]} = 1;
        }
    }

    foreach my $n (@$machines)
    {
        if (!$reachable{$n})
        {
            push @unreachable, $n;
        }
    }
    if (@unreachable)
    {
        return (\@unreachable);
    }
}

#--------------------------------------------------------------------------------

=head3    cpSSHFiles

        Returns the locations of authorized keys in /csminstall.

        Arguments:
                none
        Returns:
                ($authorizedKeys1, $authorizedKeys2)
        Globals:
                none
        Error:
                returns 1 or exits on message E value
        Example:
                $rc = NetworkUtils->cpSSHFiles();

        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub cpSSHFiles
{
    my ($cmd, $rc);
    NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                              'V', 'IMsgCopySSHKeys');
    my $SSHdir = "/csminstall/csm/config/.ssh";
    if (!(-d "/csminstall/csm/config"))
    {
        $cmd = "$::MKDIR /csminstall/csm/config";
        NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                  'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            NodeUtils->message('E', 'EMsgNO_CreateDir',
                               "/csminstall/csm/config");
        }
    }
    if (!(-d "$SSHdir"))
    {
        $cmd = "$::MKDIR $SSHdir";
        NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                  'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', "E", 'EMsgNO_CreateDir',
                                      $SSHdir);
        }
    }
    my $home             = InstallUtils->getRootHomeDir();
    my $authorized_keys  = "$SSHdir/authorized_keys";
    my $authorized_keys2 = "$SSHdir/authorized_keys2";
    if (   !(-e "$home/.ssh/identity.pub")
        || !(-e "$home/.ssh/id_rsa.pub")
        || !(-e "$home/.ssh/id_dsa.pub"))
    {
        return 1;
    }

    $cmd = "$::COPY $home/.ssh/identity.pub $authorized_keys";
    NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils', 'V',
                              'IMsgCMD', $cmd);
    `$cmd 2>&1`;
    $rc = $? >> 8;
    if ($rc)
    {
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'E', 'EMsgCANT_RUN', $cmd, $rc);
    }

    $cmd = "$::COPY $home/.ssh/id_rsa.pub $authorized_keys2";
    NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils', 'V',
                              'IMsgCMD', $cmd);
    `$cmd 2>&1`;
    $rc = $? >> 8;
    if ($rc)
    {
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'E', 'EMsgCANT_RUN', $cmd, $rc);
    }

    $cmd = "$::CAT $home/.ssh/id_dsa.pub >> $authorized_keys2";
    NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils', 'V',
                              'IMsgCMD', $cmd);
    `$cmd 2>&1`;
    $rc = $? >> 8;
    if ($rc)
    {
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'E', 'EMsgCANT_RUN', $cmd, $rc);
    }

    if (!defined $::HAMODE)
    {
        my $outref =
          NodeUtils->runrmccmd('lsrsrc-api', "-i", "-s IBM.DmsCtrl::::HAMode");
        $::HAMODE = $$outref[0];
    }
    if ($::HAMODE == 1)
      {    #on an active HAMS management server
	#get management server attribute for node on other ms
        require HAMS;
	my $inactivems = HAMS->determineInactive();
	if ($inactivems ne "")
	  {
	    NodeUtils->runcmd(
			      "/opt/csm/bin/dsh -n $inactivems /opt/csm/csmbin/remoteshell.expect -k"
			     );

                #my $auth_key = "$SSHdir/inactivems_auth";
                #my $auth_key2 = "$SSHdir/inactivems_auth2";
                NodeUtils->runcmd(
                              "/usr/bin/scp $inactivems:.ssh/\*.pub $SSHdir/.");
                NodeUtils->runcmd(
                             "$::CAT $SSHdir/identity.pub >> $authorized_keys");
                NodeUtils->runcmd(
                              "$::CAT $SSHdir/id_rsa.pub >> $authorized_keys2");
                NodeUtils->runcmd(
                              "$::CAT $SSHdir/id_dsa.pub >> $authorized_keys2");
                NodeUtils->runcmd("$::RM -f $SSHdir/\*.pub");
	  }
      }
    
    if (!(-e "$authorized_keys") || !(-e "$authorized_keys2"))
    {
        return 1;
    }
    return ($authorized_keys, $authorized_keys2);
}

#--------------------------------------------------------------------------------

=head3    getRemoteShell

        Get the path of the remote shell if avialble.

        Arguments:
                none
        Returns:
                path of the remote shell
        Globals:
                none
        Error:
                1 - couldn't connect to remote shell
        Example:
                $::REMOTE_SHELL = NetworkUtils->getRemoteShell();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getRemoteShell
{

    # set local scope because we only want classes on the mgmt svr
    $ENV{'CT_MANAGEMENT_SCOPE'} = 1;

    # todo: remove when lsrsrc-api converts to above
    $ENV{'CT_SESSION_SCOPE'} = 1;
    if (NodeUtils->isMgmtSvr())
    {    #only defined on managment server
        if (!$ENV{'DSH_REMOTE_CMD'})
        {
            my $outref =
              NodeUtils->runrmccmd('lsrsrc-api', "-i",
                                   "-s IBM.DmsCtrl::::RemoteShell");
            return $$outref[0];
        }
        else
        {
            return $ENV{'DSH_REMOTE_CMD'};
        }
    }
    return 1;
}

#--------------------------------------------------------------------------------

=head3    getRootHomeDir

        Get the path the root user home directory from /etc/passwd.

        Arguments:
                none
        Returns:
                path to root user home directory.
        Globals:
                none
        Error:
                none
        Example:
                $myHome = NetworkUtils->getRootHomeDir();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getRootHomeDir
{
    my @root = split ':', (`/bin/grep ^root /etc/passwd 2>&1`);
    my $home = $root[5];
    return $home;
}

#--------------------------------------------------------------------------------

=head3    getSetupRemoteShell

        Get the value of SetupRemoteShell in CSM database.

        Arguments:
                none
        Returns:
                value from database
        Globals:
                none
        Error:
                undefined
        Example:
                $::SETUP_REMOTE_SHELL =
                    NetworkUtils->getSetupRemoteShell();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub getSetupRemoteShell
{

    # set local scope because we only want classes on the mgmt svr
    $ENV{'CT_MANAGEMENT_SCOPE'} = 1;

    # todo: remove when lsrsrc-api converts to above
    $ENV{'CT_SESSION_SCOPE'} = 1;
    if (NodeUtils->isMgmtSvr())
    {    #only defined on managment server
        my $outref =
          NodeUtils->runrmccmd('lsrsrc-api', "-i",
                               "-s IBM.DmsCtrl::::SetupRemoteShell");
        return $$outref[0];
    }
}

#--------------------------------------------------------------------------------

=head3    installSSHFiles

        Installs SSH files

        Arguments:
                $filename_of_authorized_keys
                $filename_of_authorized_keys2
        Returns:
                $::OK
        Globals:
                none
        Error:
                undefined
        Example:
                NetworkUtils->installSSHFiles ( $authorized_keys,
                                                $authorized_keys2);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub installSSHFiles
{
    my ($class, $authorized_keys, $authorized_keys2) = @_;
    my ($cmd, $rc);
    my $home                  = InstallUtils->getRootHomeDir();
    my $dest_dir              = "$home/.ssh";
    my $dest_authorized_keys  = "$dest_dir/authorized_keys";
    my $dest_authorized_keys2 = "$dest_dir/authorized_keys2";

    umask(0077);
    if (!-d "$dest_dir")
    {

        # create a local directory
        $cmd = "$::MKDIR -p $dest_dir";
        $rc  = system("$cmd");
        if ($rc >> 8)
        {

            # Could not create $::CSMINSTDIR directory.
            NodeUtils->message('E', 'EMsgNO_CreateDir', $dest_dir);
            NodeUtils->messageFromCat(
                                      'csmInstall.cat',   $::MSGMAPPATH,
                                      'csminstall',       "E",
                                      'EMsgNO_CreateDir', $dest_dir
                                      );
            return $::NOK;
        }
        else
        {
            chmod 0700, $dest_dir;
        }
    }

    umask(0133);
    if (-e "$dest_authorized_keys")
    {
        $cmd = "$::CAT $authorized_keys >> $dest_authorized_keys";
        NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                  'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E', 'EMsgCANT_RUN', $cmd,
                                      $rc);
        }
    }
    else
    {

        #just copy the keys
        my $cmd = "$::COPY $authorized_keys $dest_authorized_keys";
        NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                  'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E', 'EMsgCANT_RUN', $cmd,
                                      $rc);
        }
        chmod 0644, $dest_authorized_keys;
    }
    if (-e "$dest_authorized_keys2")
    {
        $cmd = "$::CAT $authorized_keys2 >> $dest_authorized_keys2";
        NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                  'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E', 'EMsgCANT_RUN', $cmd,
                                      $rc);
        }
    }
    else
    {

        #just copy the keys
        my $cmd = "$::COPY $authorized_keys2 $dest_authorized_keys2";
        NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                  'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E', 'EMsgCANT_RUN', $cmd,
                                      $rc);
        }
        chmod 0644, $dest_authorized_keys2;
    }
}

#--------------------------------------------------------------------------------

=head3    rmNodesKnownHosts

        Removes SSH known hosts

        Arguments:
                \@node_hostnames
                 file name (optional)
        Returns:
                1 - error
                0 - success
        Globals:
                none
        Error:
                1
        Example:
                NetworkUtils->rmNodesKnownHosts(\@nodes);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub rmNodesKnownHosts
{
    my ($class, $ref_nodes, $tmpKnownHosts) = @_;
    my @node_hostnames = @$ref_nodes;
    my $home           = InstallUtils->getRootHomeDir();
    my @all_names;
    my $return = 0;
    foreach my $n (@node_hostnames)
    {
        my @parts = split /\./, $n;
        my $name = shift @parts;
        push @all_names, $name;
        foreach my $ext (@parts)
        {
            $name = "$name.$ext";
            push @all_names, $name;
        }

        #get the IP
        my $ip = inet_ntoa(inet_aton($n));
        push @all_names, $ip;
    }

    #create the sed command
    my $sed = "/bin/sed -e ";
    $sed .= "\"";
    foreach my $n (@all_names)
    {
        $sed .= "/^$n\[,| ]/d; ";
    }
    chop $sed;    #get rid of last space
    $sed .= "\"";
    my $sed2 = $sed;
    my $file = "$home/.ssh/$$";
    while (-e $file)
    {
        $file = InstallUtils->CreateRandomName($file);
    }
    if (defined $tmpKnownHosts)
    {             #use a temporary known hosts
        if (-e $tmpKnownHosts)
        {
            $sed .= " $tmpKnownHosts";
            $sed .= " > $file";
            my $printsed = $sed;
            $printsed =~ s/"//g;    #"
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $printsed);
            `$sed 2>&1`;
            my $rc = $? >> 8;
            if ($rc)
            {
                $return = 1;
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'E', 'EMsgCANT_RUN', $printsed, $rc);
            }
            my $cp = "$::MV $file $tmpKnownHosts";
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cp);
            `$cp 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                $return = 1;
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                   'csminstall', 'E', 'EMsgCANT_RUN', $cp, $rc);
            }

            # system("$::RM -f $file");
        }
    }
    else
    {    #use real known hosts
        if (-e "$home/.ssh/known_hosts")
        {
            $sed .= " $home/.ssh/known_hosts";
            $sed .= " > $file";
            my $printsed = $sed;
            $printsed =~ s/"//g;    #"
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $printsed);
            `$sed 2>&1`;
            my $rc = $? >> 8;
            if ($rc)
            {
                $return = 1;
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                             'csminstall', 'E', 'EMsgCANT_RUN', $printsed, $rc);
            }
            my $cp = "$::MV $file $home/.ssh/known_hosts";
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cp);
            `$cp 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                $return = 1;
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                   'csminstall', 'E', 'EMsgCANT_RUN', $cp, $rc);
            }

            #system("$::RM -f $file");
        }
        if (-e "$home/.ssh/known_hosts2")
        {
            $sed2 .= " $home/.ssh/known_hosts2";
            $sed2 .= " > $file";
            my $printsed2 = $sed2;
            $printsed2 =~ s/"//g;    #"
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $printsed2);
            `$sed2 2>&1`;
            my $rc = $? >> 8;
            if ($rc)
            {
                $return = 1;
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                            'csminstall', 'E', 'EMsgCANT_RUN', $printsed2, $rc);
            }
            my $cp = "$::MV $file $home/.ssh/known_hosts2";
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cp);
            `$cp 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                $return = 1;
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                   'csminstall', 'E', 'EMsgCANT_RUN', $cp, $rc);
            }

            #system("$::RM -f $file");
        }
    }
    return $return;
}

#--------------------------------------------------------------------------------

=head3    setupAIXRSH

        Setup RSH on AIX Managment Server

        Arguments:
                $fullHostNameofManagmentServer
        Returns:
                none
        Globals:
                none
        Error:
                none
        Example:
                NetworkUtils->setupAIXRSH($::csm_server);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub setupAIXRSH
{
    my ($class, $ms, $inactivems) = @_;
    umask(0177);
    my $home = InstallUtils->getRootHomeDir();
    my ($cmd, $rc);
    $cmd = "/bin/grep \"^$ms root\" $home/.rhosts";
    `$cmd 2>&1`;
    $rc = $? >> 8;
    if ($rc)
    {
        $cmd = "/bin/echo $ms root >> $home/.rhosts";
        NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                  'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;

        if ($rc)
        {
            NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E', 'EMsgCANT_RUN', $cmd,
                                      $rc);
        }
    }
    if (defined $inactivems)
    {
        $cmd = "/bin/grep \"^$inactivems root\" $home/.rhosts";
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            $cmd = "/bin/echo $inactivems root >> $home/.rhosts";
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cmd);
            `$cmd 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }
        }
    }
    chmod 0600, "$home/.rhosts";
}

#--------------------------------------------------------------------------------

=head3    setupLinuxRSH

        Setup RSH on Linux Mangement Server

        Arguments:
                $fullHostNameofManagmentServer
        Returns:
        none
        Globals:
                none
        Error:
                undefined
        Example:
                NetworkUtils->setupLinuxRSH($::csm_server);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub setupLinuxRSH
{
    my ($class, $ms, $inactivems) = @_;
    umask(0177);
    my $home = InstallUtils->getRootHomeDir();
    my ($cmd, $rc);

    # SuSE Changes
    # Setup rsh in inetd on SuSE
    my $distroname = NodeUtils->get_DistributionName;
    if (($distroname =~ /SuSE/) || ($distroname =~ /SLES/))
    {
        `/bin/rpm -q inetd 2>&1`;
        if ($? >> 8)
        {    #setup rsh by xinetd
            $cmd =
              "/bin/sed -e \"s/disable.*=.*yes/disable = no/\" /etc/xinetd.d/rsh";
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cmd);
            my @rsh = `$cmd 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }
            open(RSH, ">/etc/xinetd.d/rsh")
              or NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                 'csminstall', "E", 'EMsgCANT_WRITE_FILE', "/etc/xinetd.d/rsh");
            print RSH @rsh;
            close RSH;
            chmod 0600, "/etc/xinetd.d/rsh";
            $cmd = "/etc/init.d/xinetd restart";
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cmd);
            `$cmd 2>&1`;
            $rc = $? >> 8;

            if ($rc)
            {
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }
            if (-e "/etc/securetty")
            {
                $cmd = "/bin/grep rsh /etc/securetty";
                `$cmd 2>&1`;
                $rc = $? >> 8;
                if ($rc)
                {
                    `/bin/echo rsh >> /etc/securetty`;
                }
            }
        }
        else
        {
            $cmd = "/bin/sed -e \"s/#.*shell/shell/\" /etc/inetd.conf";
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cmd);
            my @rsh = `$cmd 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }
            open(RSH, ">/etc/inetd.conf")
              or NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                   'csminstall', "E", 'EMsgCANT_WRITE_FILE', "/etc/inetd.conf");
            print RSH @rsh;
            close RSH;
            chmod 0600, "/etc/inetd.conf";
            $cmd = "/etc/rc.d/inetd restart";
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cmd);
            `$cmd 2>&1`;
            $rc = $? >> 8;

            if ($rc)
            {
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }
        }
    }
    else
    {
        $cmd =
          "/bin/sed -e \"s/disable.*=.*yes/disable = no/\" /etc/xinetd.d/rsh";
        NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                  'V', 'IMsgCMD', $cmd);
        my @rsh = `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E', 'EMsgCANT_RUN', $cmd,
                                      $rc);
        }
        open(RSH, ">/etc/xinetd.d/rsh")
          or NodeUtils->messageFromCat(
                                       'csmInstall.cat',
                                       $::MSGMAPPATH,
                                       'csminstall',
                                       "E",
                                       'EMsgCANT_WRITE_FILE',
                                       "/etc/xinetd.d/rsh"
                                       );
        print RSH @rsh;
        close RSH;
        chmod 0600, "/etc/xinetd.d/rsh";
        $cmd = "/sbin/service xinetd restart";
        NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                  'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;

        if ($rc)
        {
            NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E', 'EMsgCANT_RUN', $cmd,
                                      $rc);
        }
        if (-e "/etc/securetty")
        {
            $cmd = "/bin/grep rsh /etc/securetty";
            `$cmd 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                `/bin/echo rsh >> /etc/securetty`;
            }
        }
    }

    # End of SuSE change

    $cmd = "/bin/grep \"^$ms root\" $home/.rhosts";
    `$cmd 2>&1`;
    $rc = $? >> 8;
    if ($rc)
    {
        $cmd = "/bin/echo $ms root >> $home/.rhosts";
        NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH, 'NodeUtils',
                                  'V', 'IMsgCMD', $cmd);
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'E', 'EMsgCANT_RUN', $cmd,
                                      $rc);
        }
    }
    if (defined $inactivems)
    {
        $cmd = "/bin/grep \"^$inactivems root\" $home/.rhosts";
        `$cmd 2>&1`;
        $rc = $? >> 8;
        if ($rc)
        {
            $cmd = "/bin/echo $inactivems root >> $home/.rhosts";
            NodeUtils->messageFromCat('nodecmds.cat', $::MSGMAPPATH,
                                      'NodeUtils', 'V', 'IMsgCMD', $cmd);
            `$cmd 2>&1`;
            $rc = $? >> 8;
            if ($rc)
            {
                NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                  'csminstall', 'E', 'EMsgCANT_RUN', $cmd, $rc);
            }
        }
    }
    chmod 0600, "$home/.rhosts";
}

#--------------------------------------------------------------------------------

=head3    setupRemoteShellforFullInstall

        Sets up the remote shell across a list of nodes.

        Arguments:
                Reference to an array of nodeNames
        Returns:
                1 - error
                0 - success
        Globals:
                $::REMOTE_SHELL         - shell to configure
                $SETUP_REMOTE_SHELL     - bool: 1 == setup; 0 = don't setup
        Error:
                1
        Example:
                if (NetworkUtils->setupRemoteShellforFullInstall(\@::NODELIST) != 0) {
                    blah;
                }
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub setupRemoteShellforFullInstall
{
    my $rc = 0;

    my ($class, $ref_nodes) = @_;
    my @node_hostnames = @$ref_nodes;

    # SETUP_REMOTE_SHELL controls whether or not to set up the remote shell
    # on the nodes.  It is used when creating the config_file.
    #     0=do not set up remote shell
    #     1=set up whichever remote shell is listed in $::REMOTE_SHELL
    $::SETUP_REMOTE_SHELL = InstallUtils->getSetupRemoteShell();

    # If the SetupRemoteShell attribute is set to 0 (or null) in DMSCTRL,
    # don't do any remote shell setup.
    if (!$::SETUP_REMOTE_SHELL)
    {
        return $rc;
    }

    $::REMOTE_SHELL = InstallUtils->getRemoteShell();

    # If the REMOTE_SHELL is set to 1, there was an error getting it from
    # the dmsctrl.  Don't bother setting up the remote shell.
    if ($::REMOTE_SHELL == 1)
    {
        return $::NOK;
    }

    (my $cmd_basename = $::REMOTE_SHELL) =~ s:^.*/::;
    if ($cmd_basename eq "rsh")
    {
        $::SETUP_REMOTE_SHELL = 1;
    }
    else    # Attempt to set up ssh
    {
        if (InstallUtils->supportedSSHversion())
        {

            # Check permissions on home directory
            $rc = InstallUtils->checkHOMEforSSHperm();
            if ($rc)
            {
                $::SETUP_REMOTE_SHELL = 0;
                return $rc;
            }

            # Setup ssh keys
            NodeUtils->runcmd("/opt/csm/csmbin/remoteshell.expect -k", 0);

            # Copy ssh keys to /csminstall
            $rc = InstallUtils->cpSSHFiles();
            if ($rc == 1)
            {
                $::SETUP_REMOTE_SHELL = 0;
                return $rc;
            }

            # Clean up the Known Hosts file for each node so that when the
            # nodes are reinstalled, the hosts files will be regenerated.
            #$rc = InstallUtils->rmNodesKnownHosts(\@node_hostnames);
            #the rmNodesKnownHosts step was moved to GatherSSHHostKeys

            # ssh was set up properly, so set it up on the nodes.
            $::SETUP_REMOTE_SHELL = 1;

            # Add RemoteShell to config_file
        }
    }

    return $rc;
}

#--------------------------------------------------------------------------------

=head3    supportedSSHversion

        Check if SSH version is supported by CSM install.

        Arguments:
                Requires proper valued in $::REMOTE_SHELL
        Returns:
                1 - remote shell is a supported version 
                0 - unsupported vesion
        Globals:
                $::REMOTE_SHELL
        Error:
                0
        Example:
                my $sshOk = 
                    NetworkUtils->supportedSSHversion();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub supportedSSHversion
{

    #check if its a supported version of OpenSSH
    my $ssh_check = "$::REMOTE_SHELL -V";
    my @ssh_res   = `$ssh_check 2>&1`;
    my $ssh_rc    = $? >> 8;
    if (!$ssh_rc)
    {    #successful
        if ($ssh_res[0] =~ m/OpenSSH.(\d)\.\d/)
        {    #we can set this up
            if ($1 >= 2)
            {
                if (wantarray)
                {
                    return (1, $1);
                }
                else
                {
                    return 1;
                }
            }
        }
        else
        {
            return 0;
        }
    }
}

#--------------------------------------------------------------------------------

=head2    TFTP USER Support

=cut

#--------------------------------------------------------------------------------

=head3    check4InstalledTftpServers

	Checks the current install for tftp rpms

        Arguments:
                $stars
        Returns:
               @tftpRpms  - array of rpms found.
        Globals:
                $::TFTPUSER
		$default_tftp_server_name
        Error:
                error count
        Example:
                 NetworkUtils->check4InstalledTftpServers();
        Comments:
                none


=cut

#--------------------------------------------------------------------------------

sub check4InstalledTftpServers
{

    # check if there are any tftp rpms installed
    my ($class, $stars) = @_;
    my $cmd = "rpm -qa | grep tftp";

    # check and see if any tftp rpms are installed:
    my @tftpRpms = NodeUtils->runcmd($cmd, -1);
    if ($::RUNCMD_RC eq 0)
    {

        # first check:  If our default tftp server is installed, then
        # we can just skip:
        foreach (@tftpRpms)
        {
            my $temp = $_;
            if (grep (/^$temp$/, @::default_tftp_server_name))
            {

                # the default was found:
                MessageUtils->message('V', 'IMsgRPM_ALREADY_INSTALLED',
                                      @::default_tftp_server_name);
                return (1, @tftpRpms)
                  ;    # the 1 means that it is the default server
            }
        }
        if ($stars)
        {
            print "\n";
            print "*" x 80 . "\n";
        }

        # how many of them are actually installed?
        if ($#tftpRpms ne -1)
        {
            MessageUtils->message('V', 'IMsgRPM_ALREADY_INSTALLED',
                                  @::default_tftp_server_name);
        }
        else
        {
            MessageUtils->message('V', 'IMsgRPM_ALREADY_INSTALLED',
                                  @::default_tftp_server_name);
        }
        foreach (@tftpRpms) { print "\t$_\n"; }
    }
    return (0, @tftpRpms);
}

#--------------------------------------------------------------------------------

=head3    chTftpUserPermissions

        Change permissions on TFTP user files.
	Support routine for calls to find.

        Arguments:
                none
        Returns:
               none
        Globals:
                $::TFTPUSER
    		$::TFTPGROUP
        Error:
                error count
        Example:
		 find(\&chTftpUserPermissions, $dir);
        Comments:
 		none

=cut

#--------------------------------------------------------------------------------

sub chTftpUserPermissions
{

    # these guys seem to be in the wrong place.
    $::TFTPUSER  = "tftpd";
    $::TFTPGROUP = $::TFTPUSER;

    if (NetworkUtils->testTftpUser == 0) { NetworkUtils->createTftpUser; }

    my $filename = $File::Find::name;

    if (-f $filename)
    {
        NodeUtils->runcmd("chmod $::filePermissions $filename");
    }
    elsif (-d $filename)
    {
        NodeUtils->runcmd("chmod $::dirPermissions $filename");
    }
    else
    {
        print "---> $filename is not a file nor a directory.\n";
        return;
    }

    # change owner and group:
    NodeUtils->runcmd("chown $::TFTPUSER $filename");
    NodeUtils->runcmd("chgrp $::TFTPGROUP $filename");

}

#--------------------------------------------------------------------------------

=head3    configureTftp

        Configure the default tftp server 

        Arguments:
                none
        Returns:
                error count or 0 (success)
        Globals:
                none
        Error:
                error count
        Example:
                 my $eCount = NetworkUtils->configureTftp();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub configureTftp
{

    # Configure inetd in SuSE
#   my $distro = NodeUtils->distribution();
#   if ($distro =~ /SuSE/ || $distro =~ /SLES/)
#   if (($distro =~ /SuSE/ || $distro =~ /SLES/)&&(!($distro =~ /SLES 9/)))
#   my $inetd = "/etc/init.d/inetd";
#    if (-f $inetd)
#    {
#        my $inetderr = configureTftp_inetd();
#        return $inetderr;
#    }

    my $inetd = "/etc/init.d/inetd";
    my $xinetd       = "/etc/init.d/xinetd";
    my $tftpConf     = "/etc/xinetd.d/tftp";
    my $xinetdbackup = "/tmp/tftp.bak";
    my $errors       = 0;
    my ($cmd, $output, $rc);

    unless (-f $xinetd)
    {
        if (-f $inetd)
	{
	        my $inetderr = configureTftp_inetd();
	        return $inetderr;
        }else{
		NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  "E", 'EMsgXINETD_MISSING', "$xinetd");
        	$errors++;
	}	
    }

    # save old file to tmp
    if (-f "$tftpConf")
    {
        $cmd    = "$::COPY -f $tftpConf $xinetdbackup";
        $output = NodeUtils->runcmd($cmd, 0);
        $rc     = $::RUNCMD_RC;
        if ($rc)
        {
            NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH,
                                      'csminstall', 'I', 'IMsgShow_Output',
                                      $output);
            $errors++;
        }
    }

    # Now, configure our file:
    open(FILE, ">$tftpConf")
      or die NodeUtils->messageFromCat(
                                       'csmInstall',          $::MSGMAPPATH,
                                       'csminstall',          'E1',
                                       'EMsgCANT_WRITE_FILE', $tftpConf
                                       );
    my @fC = <<EOF;
# description: The tftp server serves files using the trivial file transfer \
#       protocol.  The tftp protocol is often used to boot diskless \
#       workstations, download configuration files to network-aware printers, \
#       and to start the installation process for some operating systems.
service tftp
{
        socket_type             = dgram
        protocol                = udp
        wait                    = yes
        user                    = root
        server                  = /usr/sbin/in.tftpd
        server_args             = -vvv -r blksize -p -u tftpd -s /tftpboot
}
EOF
    foreach (@fC) { print FILE "$_"; }
    close(FILE);
    $errors =
      NetworkUtils->restartTftp($errors, "xinetd");    # restart tftp server.
    return $errors;
}

#--------------------------------------------------------------------------------

=head3    configureTftp_inetd

        Modify /etc/inetd.conf for tftp

        Arguments:
                none
        Returns:
                error count
        Globals:
                none
        Error:
                error count or 0 (success)
        Example:
                 my $inetderr = configureTftp_inetd();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub configureTftp_inetd
{

    my $inetdconf   = "/etc/inetd.conf";
    my $inetdbackup = "/tmp/tftp_inetd.bak";
    my $errors      = 0;
    my ($cmd, $output, $rc, $s);

    unless (-f "$inetdconf") { return ($errors++); }

    # backup inetd.conf
    $cmd    = "$::COPY -f $inetdconf $inetdbackup ";
    $output = NodeUtils->runcmd($cmd, 0);
    $rc     = $::RUNCMD_RC;
    if ($rc)
    {
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'I', 'IMsgShow_Output', $output);
        $errors++;
    }

    # now edit the inetd.conf file:
    $s = NodeUtils->readFile($inetdconf);

    #comment tftp entry
    $s =~ s/^\s*(tftp\s+[\s\S]*$)//m;
    $s = $s
      . "\ntftp    dgram   udp     wait    root    /usr/sbin/in.tftpd in.tftpd -r blksize -vvv -p -u tftpd -s /tftpboot\n";
    if (open(FILE, ">$inetdconf"))
    {
        print FILE ($s);
        close(FILE);
    }
    else { exit; }

    # Restart inetd to pick up the changes
    $errors =
      NetworkUtils->restartTftp($errors, "inetd");    # restart tftp server.
    return $errors;
}

#--------------------------------------------------------------------------------

=head3    createTftpUser

        Create the TFTP user to secure system directories

        Arguments:
                none
        Returns:
                none
        Globals:
                $::TFTPUSER
        Error:
                V & I Err mesages
        Example:
                 NetworkUtils->createTftpUser();
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub createTftpUser
{
    unless ($::GREP) { $::GREP = '/bin/grep' }
    my $tftpuser   = $::TFTPUSER;
    my $tftpgrp    = $::TFTPGROUP;
    my $passwdFile = "/etc/passwd";
    my $grpFile    = "/etc/group";
    my $bak        = "/tmp/csmjunk";

    # determine whether or not a tftpd user already exists:
    NodeUtils->runcmd("$::GREP -e '^$tftpuser' $passwdFile", -1);
    if ($::RUNCMD_RC eq 0)
    {

        #print "tftpd user has already been created.\n" if $::VERBOSE;
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  "V", 'IMsgUserAlreadyCreated', $tftpuser);
    }
    else
    {

        #print "Creating default tftpd user for the tftp server.\n";
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  "I", 'IMsgCreatingTFTPUser', $tftpuser);

        # create tftpd user
        #
        NodeUtils->runcmd("cp $passwdFile $bak");
        NodeUtils->runcmd(
            "echo '$tftpuser:x:6969:6969::/dev/null:/dev/null' | \
		cat $bak - > $passwdFile");

        # remove backup file
        NodeUtils->runcmd("rm $bak", -1);
    }
    NodeUtils->runcmd("$::GREP -e '^$tftpgrp' $grpFile", -1);
    if ($::RUNCMD_RC eq 0)
    {

        #print "tftpd group has already been created.\n" if $::VERBOSE;
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  "V", 'IMsgGroupAlreadyCreated', $tftpgrp);
    }
    else
    {

        #print "Creating default tftpd group for the tftp server.\n";
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  "I", 'IMsgCreatingTFTPGroup', $tftpgrp);
        NodeUtils->runcmd("cp $grpFile $bak");
        NodeUtils->runcmd(
            "echo '$tftpgrp:x:6969:' | \
		cat $bak - > $grpFile");

        # remove backup file
        NodeUtils->runcmd("rm $bak", -1);
    }
}

#--------------------------------------------------------------------------------

=head3    process_tftp

	handles installation of tftp-hpa rpm.  

	Special case for tftp:

	1.  The tftp-hpa (default csm rpm) may already be installed.
		skip and do nothing

 	2.  Another tftp server may already be installed
 		remove old tftp server and install new?
		keep old tftp server and move on?
		keep old tftp server and install new tftp server
		quit

 	3.  No tftp server RPM is installed
		just install the tftp server.

 	if process_tftp returns 1 then install the default tftp package,
 	otherwise, do not install it. (returns 0).

	Notes:

	Stuff like this should probably not be in a shared package.

=cut

#--------------------------------------------------------------------------------

sub process_tftp
{
    my $installTftp = 0;    # 0 means no, 1 means install it.
    my @vn = @::default_tftp_server_name;
    my ($cmd, $output, $rc, $returncode);
    $returncode = 0;
    my ($isDefault, @tftpRpms) = NetworkUtils->check4InstalledTftpServers(1);
    my $ans = "";
    if ($#tftpRpms eq -1)
    {

        # no tftp servers were found.  BUT, perhaps there is a
        # tarball installed.  We should check for that!
        my $running = NetworkUtils->verifyTftp();
        if ($running)
        {
            print "*" x 80 . "\n";
            NodeUtils->message('I', 'IMsgTftpProcFound', @vn);
            while (1)
            {
                NodeUtils->message('I', 'IMsgTftpOptionsMenu', @vn, @vn);
                $|   = 1;         # force a flush
                $ans = <STDIN>;
                chomp($ans);
                if ($ans eq 1 || $ans eq 2 || $ans eq 3) { last; }
                print "\n\n";
                print "*" x 80 . "\n";

            }
            if ($ans eq 3)
            {
                exit;
            }
            elsif ($ans eq 1)
            {

                # install @::default_tftp_server_name
                $installTftp = 1;    # install tftp
            }
            elsif ($ans eq 2)
            {

                # warn them if they do not install tftp...
                NodeUtils->message('I', 'IMsgChangeTftpUser', 'tftpd');
                my $d = <STDIN>;

                # do nothing and leave.
                NodeUtils->message('I', 'IMsgWILL_NOT_INSTALL',
                                   @::default_tftp_server_name);
            }

        }    # there were no tftp processes running.
        else
        {

            # so we need to install!
            $installTftp = 1;
        }
    }
    elsif ($#tftpRpms ne -1 && !$isDefault)
    {
        NodeUtils->message('I', 'IMsgTFTPInfo', @vn, @vn);

        while (1)
        {
            NodeUtils->message('I', 'IMsgTFTPOptions', @vn, @vn, @vn, @vn);
            $|   = 1;         # force a flush
            $ans = <STDIN>;
            chomp($ans);
            if ($ans eq 1 || $ans eq 2 || $ans eq 3 || $ans eq 4) { last; }
            print "\n\n";
            print "*" x 80 . "\n";

        }

        # warn them if they do not install tftp...
        unless ($ans eq 1 || $ans eq 2 || $ans eq 4)
        {
            NodeUtils->message('I', 'IMsgChangeTftpUser', 'tftpd');
            my $d = <STDIN>;
        }

        if ($ans eq 1)
        {

            # remove tftp rpms and install @::default_tftp_server_name
            NetworkUtils->rmTftpRpms(@tftpRpms);
            $installTftp = 1;    # install it!
        }
        elsif ($ans eq 2)
        {

            # install @::default_tftp_server_name
            $installTftp = 1;    # install tftp
        }
        elsif ($ans eq 3)
        {

            # do nothing and leave.
            NodeUtils->message('I', 'IMsgWILL_NOT_INSTALL',
                               @::default_tftp_server_name);
        }
        elsif ($ans eq 4)
        {
            exit;
        }
        else
        {

            # if we got here something went wrong
            print "internal error 127";
            exit;
        }
    }
    elsif (!$isDefault)
    {

        # no tftp server is installed, so install the tftp server:
        $installTftp = 1;
    }

    # the last case is if the default tftp server was already installed.
    # if this is true then nothing needs to be returned (0) because we
    # do not wish to install it again.
    return $installTftp;
}

#--------------------------------------------------------------------------------

=head3    restartTftp

    Restart xinetd to pick up changes.

        Arguments:
        	$errors
        	$service  - e.g. "inetd" or "xinetd"
        Returns:
                Number of errors
        Globals:
                none
        Error:
               Number of errors as a scalar.
        Example:
                $errors = NetworkUtils->restartTftp($errors, "inetd" );
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub restartTftp
{
    my ($class, $errors, $service) = @_;

    my $rc;

    # Restart xinetd to pick up the changes
    my $cmd = NodeUtils->service($service, "start");
    NodeUtils->runcmd($cmd, 0);
    if ($rc) { $errors++ }

    $cmd = NodeUtils->service($service, "reload");
    NodeUtils->runcmd($cmd, 0);
    if ($rc) { $errors++ }

    return $errors;
}

#--------------------------------------------------------------------------------

=head3    rmTftpRpms

        Removes uses rpm to remove TFTP RPMs

=cut

#--------------------------------------------------------------------------------

sub rmTftpRpms
{
    my ($class, @tftpRpms) = @_;

    my ($class, $output, $returncode, $ans);
    $returncode = 1;    # signal an error by default, unless things run right.

    foreach my $installedTftp (@tftpRpms)
    {
        NodeUtils->message("I", 'IMsgUNINSTALLING', $installedTftp);
        my $cmd = "$::RPMCMD -e $installedTftp --nodeps";
        $output = NodeUtils->runcmd($cmd, 0);
        $returncode = $::RUNCMD_RC if ($::RUNCMD_RC > $returncode);
    }
    return ($returncode);
}

#--------------------------------------------------------------------------------

=head3    secureFilePermissions

        Configure the default tftp server 

        Arguments:
                $directory   - the root directory
                $dir-perm    - permissions for dirs and subdirs
                $file-perm   - permissions for files
                
        Returns:
                none
        Globals:
                none
        Error:
                none
        Example:
                 NetworkUtils->secureFilePermissions ( $directory,
                                                       $dir-perm,
                                                       $file-perm);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub secureFilePermissions
{
    my ($class, $dir, $dirPerm, $filePerm,$disable_recurtive) = @_;

    # set these variables global, since &wanted does not
    # take parameters (see find routine in perl docs).

    $::dirPermissions  = $dirPerm;
    $::filePermissions = $filePerm;

    if($disable_recurtive==0){
	find(\&chTftpUserPermissions, $dir);
    }else{
	$File::Find::name = $dir;
	&chTftpUserPermissions;
    }

}

#--------------------------------------------------------------------------------

=head3   testTftpUser

        Tests that the tftp user exists. 

        Arguments:

        Returns:
                0 = does not exit
                1 = does exits
                errors
        Globals:
                $::TFTPUSER
                $passwdFile
        Error:
                none
        Example:
                if ( NetworkUtils->testTftpUser ) { blah;};
        Comments:
                ATTN:
                set $tftpuser and $passwdFile in this sub.
                the tftp stuff might be better in a security package.

=cut

#--------------------------------------------------------------------------------

sub testTftpUser
{
    my $tftpuser   = $::TFTPUSER;
    my $passwdFile = "/etc/passwd";

    # determine whether or not a tftpd user already exists:
    NodeUtils->runcmd("$::GREP -e '^$tftpuser' $passwdFile", -1);
    if ($::RUNCMD_RC eq 0)
    {
        return 1;
    }

    return 0;
}

#--------------------------------------------------------------------------------

=head3    verifyTftp

        Check that tftp server is running.  If a value of 1 is passed
        in as an argument, then the routine will try to restart tftp if
        it isn't running.

        Arguments:
                optional argument "1"

        Returns:
                0 = not running
                1 = running
                errors
        Globals:
                none
        Error:
                return value of restart routine
        Example:
                my $running = NetworkUtils->verifyTftp(1);
        Comments:
                none

=cut

#--------------------------------------------------------------------------------

sub verifyTftp
{

    # if the value 1 is passed in then we should try to restart tftp if
    # it is not running.
    my ($class, $restart) = @_;

    my $errors  = 0;
    my $running = 0;
    my $cmd;

    NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                              'V', 'IMsgCheckingTftp');

    unless ($::LSOF) { $::LSOF = '/usr/sbin/lsof' }
    $cmd = "$::LSOF -i UDP:tftp";
    NodeUtils->runcmd($cmd, -1);

    if ($::RUNCMD_RC)
    {

        # There is no tftp server running on this machine
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'I', 'IMsgTftpNotRunning', 'tftp');
    }
    else
    {

        # A tftp server is up and running.
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'V', 'IMsgRunning', 'tftp');
        $running = 1;
    }

    # try to restart it
    if ($restart && $::RUNCMD_RC)
    {

        # trying to restart tftp server
        NodeUtils->messageFromCat('csmInstall.cat', $::MSGMAPPATH, 'csminstall',
                                  'I', 'IMsgRestart', 'tftp');
        my $distro = NodeUtils->distribution();

        # restart tftp server.
	my $inetd = "/etc/init.d/inetd";
	my $xinetd = "/etc/init.d/xinetd";
#       if (($distro =~ /SuSE/ || $distro =~ /SLES/)&&(!($distro =~ /SLES 9/)))
	if (-f $xinetd)       
        {
            $errors = NetworkUtils->restartTftp($errors, "xinetd");
        }
        elsif (-f $inetd)
        {
            $errors = NetworkUtils->restartTftp($errors, "inetd");
        }
        $cmd = "$::LSOF -i UDP:tftp";
        NodeUtils->runcmd($cmd, -1);
        if ($::RUNCMD_RC)
        {
            NodeUtils->messageFromCat(
                                      'csmInstall.cat',       $::MSGMAPPATH,
                                      'csminstall',           'I',
                                      'IMsgStillCantRestart', 'tftp'
                                      );

        }
        else
        {
            $running = 1;
        }
    }
    return $running;
}

#--------------------------------------------------------------------------------

=head3    validate_ip

        Check that tftp server is running.  If a value of 1 is passed
        in as an argument, then the routine will try to restart tftp if
        it isn't running.

        Arguments:
                hostname

        Returns:
                -1 = errors
                ipaddress 
        Globals:
                none
        Error:
                return ip address for the hostname
        Example:
                my $ipaddr = NetworkUtils->validate_ip($hostname);
        Comments:
                none
=cut

#--------------------------------------------------------------------------------

sub validate_ip
{
    my $class = shift;
    my $host  = shift;
    my ($work, $res_ip_addr);

    $work = inet_aton($host);
    if (!defined($work))
    {
        $res_ip_addr = -1;
    }
    else
    {
        $res_ip_addr = inet_ntoa($work);
    }

    return $res_ip_addr;
}

#--------------------------------------------------------------------------------

=head3    ssh_keyscan


        Arguments:
                node name

        Returns:
                @keys array

        Globals:
                none
        Error:
              @keys is empty  
        Example:
                my @keys = NetworkUtils->ssh_keyscan($hostname);
        Comments:
                none
=cut

#--------------------------------------------------------------------------------

sub ssh_keyscan
{
  my ($class, $node) = @_;
  require Socket;
  my $keyscan = "/usr/bin/ssh-keyscan";
  my @names;
  my @keys;
  my %k;

  my $res =0; #used to see if hostname is resolvable

  # get short names
  my @parts = split /\./, $node;
  my $name = shift @parts;
  NodeUtils->tryHost($name, \$res);
  if($res){
    push @names, $name;
  }
  foreach my $ext (@parts)
    {
      $name = "$name.$ext";
      NodeUtils->tryHost($name, \$res);
      if($res){
	push @names, $name;
      }
    }

  #real name
  NodeUtils->tryHost($node, \$res); #tryHost sets $resolved to zero at startup
  if($res){
    push @names, $node;
  }

  #get the IP
  my $ip = Socket::inet_ntoa(Socket::inet_aton($node));
  push @names, $ip;
  my @otherips = NodeUtils->runcmd("CT_MANAGEMENT_SCOPE=3 /usr/bin/lsrsrc-api -i -s IBM.NetworkInterface::\"NodeNameList IN ('$node')\"::IPAddress",-1);
  if(scalar @otherips >1){ #we were able to contact the node over RSCT
    foreach my $oip (@otherips){
      push @names, $oip;
      my $host = NodeUtils->tryHost($oip);
      if($host ne $oip){
	push @names, $host;
      }
    }
  }

  #make the name list
  my %nhash;
  foreach my $n (@names){ $nhash{$n}=1; } #remove duplicate names
  my $namestring = join ",", (keys %nhash);

  #get the hostkeys
  my $string = join " ", (keys %nhash);
  $::upd_debug && print "Debug running: $keyscan -t rsa1,rsa,dsa $string\n";
  my @output= `$keyscan -t rsa1,rsa,dsa $string 2>/dev/null`;
  chomp @output;

  foreach my $line (@output){
    if($line !~ m/^\#/){
      my ($hostname, $rest) = split " ", $line, 2;
      $k{$rest}{$hostname}=1;
    }
  }
 
  foreach my $i (keys %k){ #this is the ssh host key
    my $goodkey=0;
    foreach my $name (keys %{$k{$i}}){
      if(!$goodkey && defined $nhash{$name}){ $goodkey=1; } #this is a hostname we are looking for
    }
    if($goodkey){ push @keys, "$namestring $i\n"; }
  }
  
  return @keys;
}


#--------------------------------------------------------------------------------

=head3  update_known_hosts


        Arguments:
                array of nodes

        Returns:


        Globals:
                $::SETUP_REMOTE_SHELL $::REMOTE_SHELL
        Error:

        Example:
                NetworkUtils->update_known_hosts(\@nodes);
        Comments:
                none
=cut

#--------------------------------------------------------------------------------

sub update_known_hosts 
{
  my ($class, $machines) = @_;
  
  my @nodes = @$machines;
  if($::REMOTE_SHELL !~ /ssh/){
    return 1;
  }
  
  my ($supported, $sshversion) =  InstallUtils->supportedSSHversion();
  if(!$supported){
    exit 0;
  }


  if($sshversion < 3){
    InstallUtils->rmNodesKnownHosts(\@nodes);
    foreach my $n (@nodes){
      my $cmd = "/opt/csm/csmbin/remoteshell.expect -n $n";
      system($cmd);
    }
  }
  else{ #ssh 3 or higher, use ssh-keyscan
    my $home           = InstallUtils->getRootHomeDir();
    my $known_hosts = "$home/.ssh/known_hosts";
    if(! -d "$home/.ssh"){ mkdir "$home/.ssh", 0700; }
    if(-e "/var/opt/csm/known_hosts_debug"){ $::DEBUG=1; }
    if($::DEBUG){
      #DEBUG
      if(! -e "/tmp/known_hosts_debug/"){
	mkdir("/tmp/known_hosts_debug");
      }
      my $time = time();
      my $orig_file = "/tmp/known_hosts_debug/$time";
      my $file = $orig_file;
      my $count = 1;
      while (-e $file){
	$file = $orig_file . "_$count"; #append count
	$count++;
      }
      open (DEBUG_FILE, ">$file") or die "cannot open $file for writing: $!";
      print DEBUG_FILE "gatherSSHHostkeys\n\n";
      #END DEBUG
    }

    #run ssh-keyscan
    my @newkeys;
    my @goodnodes; #nodes that are reachable and have host keys
    foreach my $n (@nodes){
      my @keys = NetworkUtils->ssh_keyscan($n);
      if(scalar @keys > 0){
	push @newkeys, @keys;
	push @goodnodes, $n;
      }
    }
    if(scalar @newkeys > 0){
      # First it creates an exclusive, blocking kernel lock on a file. 
      # This will only let one execution run at a time, the rest will have to wait until its finished. 
      use Fcntl qw(:flock);
      my $file = "/var/opt/csm/.CSMgatherSSHHostkeys_lock"; #lock a file in /var so it doesn't need to be removed
      if(!open(SEM,">$file")) {
	#die "cannot open file $file";
	NodeUtils->message( 'E2', 'EMsgCANT_OPEN_FILE', $file, $!, ($? >> 8) );
      }

      #use an alarm to timeout if we cannot get the lock in 10 minutes
      my $cantlock =0;
      eval { 
	local $SIG{ALRM} = sub { die "alarm clock restart" };
	alarm 600;               # schedule alarm in 600 seconds 
	eval { 
	  if(!flock(SEM,LOCK_EX))
	    {
	      $cantlock=1;
	    }
	};
	alarm 0;                # cancel the alarm
      };
      alarm 0;               # race condition protection
      die if $@ && $@ !~ /alarm clock restart/; # reraise
      if($cantlock==1){ die "cannot get lock on $file\n"; }
      
      #remove old entries from known_hosts
      $::DEBUG and print DEBUG_FILE "Removing nodes from known_hosts: @goodnodes\n\n"; #DEBUG
      InstallUtils->rmNodesKnownHosts(\@goodnodes);
      
      #append new entries to known_hosts
      $::DEBUG and print DEBUG_FILE "\n\nAdding the following lines to known_hosts:\n@newkeys"; #DEBUG
      if(-e $known_hosts){
	open(HOSTS, ">>$known_hosts") or NodeUtils->message( 'E2', 'EMsgCANT_OPEN_FILE', $known_hosts, $!, ($? >> 8) );
      }
      else{
	open(HOSTS, ">$known_hosts") or NodeUtils->message( 'E2', 'EMsgCANT_OPEN_FILE', $known_hosts, $!, ($? >> 8) );
      }
      print HOSTS @newkeys;
      close HOSTS or NodeUtils->message("E2", 'EMsgCANT_CLOSE_FILE', $known_hosts, $!, ($? >> 8));
      
      close(SEM);
    }
    else{ $::DEBUG and print DEBUG_FILE "Nothing in the array newkeys\n"; } #DEBUG
    $::DEBUG and close DEBUG_FILE;
  }
 
  #update the known_hosts file on backup ms
  if (!defined $::HAMODE)
    {
      my $outref =
	NodeUtils->runrmccmd('lsrsrc-api', "-i", "-s IBM.DmsCtrl::::HAMode");
      $::HAMODE = $$outref[0];
    }
  if ($::HAMODE == 1)
    {    #on an active HAMS management server
      require HAMS;
      my $inactive_ms = HAMS->determineInactive();
      if($inactive_ms =~ m/\w/){ #if its a hostname
	my $node_string = join " ", @$machines;
	NodeUtils->runcmd("$::DSH -n $inactive_ms CSM_HAMS_CONTROL=1 /opt/csm/csmbin/gatherSSHHostkeys $node_string", 0);
      }
    }
}
#--------------------------------------------------------------------------------

1;

